/*
 * Driver for the GRACE Metafile format
 */

#include <config.h>

#include <stdio.h>
#include <string.h>

#include "defines.h"
#include "globals.h"
#include "utils.h"
#include "draw.h"
#include "device.h"
#include "mfdrv.h"

extern FILE *prstream;

int mfinitgraphics(void)
{
    /* device-dependent routines */
    devsetpen       = mf_setpen;
    devsetline      = mf_setlinestyle;
    devsetlinew     = mf_setlinewidth;
    
    devupdatecmap   = NULL;
    
    devdrawpolyline = mf_drawpolyline;
    devdrawpolygon  = mf_drawpolygon;
    devdrawarc      = mf_drawarc;
    devfillarc      = mf_fillarc;
    devputpixmap    = mf_putpixmap;
    devputtext      = mf_puttext;
    
    devleavegraphics = mf_leavegraphics;

    fprintf(prstream, "InitGraphics { }\n");
    
    return GRACE_EXIT_SUCCESS;
}

void mf_setpen(void)
{
    Pen pen;
    
    pen = getpen();
    fprintf(prstream, "SetPen { %3d %3d }\n", pen.color, pen.pattern);
}

void mf_setlinestyle(int lines)
{
    fprintf(prstream, "SetLineStyle { %3d }\n", lines);
}

void mf_setlinewidth(double linew)
{
    fprintf(prstream, "SetLineWidth { %.4f }\n", linew);
}

void mf_drawpolyline(VPoint *vps, int n, int mode)
{
    int i;
    
    fprintf(prstream, "DrawPolyline {\n");
    if (mode == POLYLINE_CLOSED) {
        fprintf(prstream, "\tClosed\n");
    } else {
        fprintf(prstream, "\tOpen\n");
    }
    for (i = 0; i < n; i++) {
        fprintf(prstream, "\t( %.4f , %.4f )\n", vps[i].x, vps[i].y);
    }
    fprintf(prstream, "}\n");
}

void mf_drawpolygon(VPoint *vps, int nc)
{
    int i;
    
    fprintf(prstream, "DrawPolygon {\n");
    for (i = 0; i < nc; i++) {
        fprintf(prstream, "\t( %.4f , %.4f )\n", vps[i].x, vps[i].y);
    }
    fprintf(prstream, "}\n"); 
}

void mf_drawarc(VPoint vp1, VPoint vp2, int a1, int a2)
{
    fprintf(prstream, "DrawArc { ( %.4f , %.4f ) ( %.4f , %.4f ) %3d %3d }\n", 
                                   vp1.x, vp1.y,   vp2.x, vp2.y, a1, a2);
}

void mf_fillarc(VPoint vp1, VPoint vp2, int a1, int a2)
{
    fprintf(prstream, "FillArc { ( %.4f , %.4f ) ( %.4f , %.4f ) %3d %3d }\n", 
                                   vp1.x, vp1.y,   vp2.x, vp2.y, a1, a2);
}

void mf_putpixmap(VPoint vp, int width, int height, char *databits, 
                             int pixmap_bpp, int bitmap_pad, int pixmap_type)
{
    int i, j, k;
    long paddedW;
    int bit;
    char buf[16];
    
    if (pixmap_bpp == 1) {
        strcpy(buf, "Bitmap");
    } else {
        strcpy(buf, "Pixmap");
    }
    fprintf(prstream, "Put%s {\n", buf);
   
    if (pixmap_type == PIXMAP_TRANSPARENT) {
        strcpy(buf, "Transparent");
    } else {
        strcpy(buf, "Opaque");
    }
    
    fprintf(prstream, "\t( %.4f , %.4f ) %dx%d %s\n", 
                           vp.x, vp.y, width, height, buf);
    if (pixmap_bpp != 1) {
        for (k = 0; k < height; k++) {
            fprintf(prstream, "\t");
            for (j = 0; j < width; j++) {
                fprintf(prstream, "%02x", (databits)[k*width+j]);
            }
            fprintf(prstream, "\n");
        }
    } else {
        paddedW = PAD(width, bitmap_pad);
        for (k = 0; k < height; k++) {
            fprintf(prstream, "\t");
            for (j = 0; j < paddedW/bitmap_pad; j++) {
                for (i = 0; i < bitmap_pad; i++) {
                    bit = bin_dump(&databits[k*paddedW/bitmap_pad + j], i, bitmap_pad);
                    if (bit) {
                        fprintf(prstream, "X");
                    } else {
                        fprintf(prstream, ".");
                    }
                }
            } 
            fprintf(prstream, "\n");
        }
    }

    fprintf(prstream, "}\n"); 
}

void mf_puttext (VPoint start, VPoint end, double size, 
                                            CompositeString *cstring)
{
    int iglyph;
    
    fprintf(prstream, "PutText {\n");
    fprintf(prstream, "\t( %.4f , %.4f ) ( %.4f , %.4f )\n", 
                                start.x, start.y, end.x, end.y); 

    iglyph = 0;
    while (cstring[iglyph].s != NULL) {
        fprintf(prstream, "\t %d %.4f %.4f %d %d \"%s\"\n", 
                            cstring[iglyph].font,
                            size * cstring[iglyph].scale,
                            size * cstring[iglyph].vshift,
                            cstring[iglyph].underline,
                            cstring[iglyph].overline,
                            cstring[iglyph].s);
        iglyph++;
    }

    fprintf(prstream, "}\n"); 
}

void mf_leavegraphics(void)
{
    fprintf(prstream, "LeaveGraphics { }\n");
}

