/*
 *  GRACE PostScript driver
 */

#include <config.h>
#include <cmath.h>

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <time.h>

#include "defines.h"
#include "globals.h"
#include "utils.h"
#include "draw.h"
#include "device.h"
#include "patterns.h"
#include "psdrv.h"
#include "patchlevel.h"
#include "protos.h"

static int curformat = DEFAULT_PS_FORMAT;

static unsigned long page_scale;
static float page_scalef;
static int page_orientation;

static int *psfont_status = NULL;

extern FILE *prstream;

static char *escape_paren(char *s);

static int ps_initgraphics(int format)
{
    int i, j;
    Page_geometry pg;
    RGB *rgb;
    
    time_t time_value;
    
    curformat = format;
    
    /* device-dependent routines */
    devsetpen = ps_setpen;
    devsetline = ps_setlinestyle;
    devsetlinew = ps_setlinewidth;
    
    devupdatecmap = NULL;
    
    devdrawpolyline = ps_drawpolyline;
    devdrawpolygon = ps_drawpolygon;
    devdrawarc = ps_drawarc;
    devfillarc = ps_fillarc;
    devputpixmap = ps_putpixmap;
    devputtext = ps_puttext;
    
    devleavegraphics = ps_leavegraphics;

    pg = get_page_geometry();
    
    page_scale = MIN2(pg.height, pg.width);
    page_scalef = (float) page_scale*72.0/pg.dpi_x;

    if (pg.height < pg.width) {
        page_orientation = PAGE_ORIENT_LANDSCAPE;
    } else {
        page_orientation = PAGE_ORIENT_PORTRAIT;
    }

    /* Font status table */
    if (psfont_status != NULL) {
        free(psfont_status);
    }
    psfont_status = (int *) malloc(number_of_fonts()*SIZEOF_INT);
    for (i = 0; i < number_of_fonts(); i++) {
        psfont_status[i] = FALSE;
    }
    
    switch (curformat) {
    case PS_FORMAT:
        fprintf(prstream, "%%!PS-Adobe-3.0\n");
        break;
    case EPS_FORMAT:
        fprintf(prstream, "%%!PS-Adobe-3.0 EPSF-3.0\n");
        break;
    default:
        errmsg("Invalid PS format");
        return GRACE_EXIT_FAILURE;
    }
    fprintf(prstream, "%%%%BoundingBox: (atend)\n");
    fprintf(prstream, "%%%%LanguageLevel: 2\n");
    fprintf(prstream, "%%%%Creator: Grace v%d.%d.%d %s\n",
                                MAJOR_REV, MINOR_REV, PATCHLEVEL, BETA_VER);

    time(&time_value);
    fprintf(prstream, "%%%%CreationDate: %s", ctime(&time_value));
    fprintf(prstream, "%%%%DocumentData: Clean7Bit\n");

    if (page_orientation == PAGE_ORIENT_LANDSCAPE) {
        fprintf(prstream, "%%%%Orientation: Landscape\n");
    } else {
        fprintf(prstream, "%%%%Orientation: Portrait\n");
    }
    
    if (curformat == PS_FORMAT) {
        fprintf(prstream, "%%%%Pages: 1\n");
        fprintf(prstream, "%%%%PageOrder: Ascend\n");
    }
    fprintf(prstream, "%%%%Title: %s\n", docname);
    fprintf(prstream, "%%%%For: %s\n", getlogin());
    fprintf(prstream, "%%%%EndComments\n");

    /* Definitions */
    fprintf(prstream, "%%%%BeginProlog\n");
    fprintf(prstream, "/m {moveto} def\n");
    fprintf(prstream, "/l {lineto} def\n");
    fprintf(prstream, "/s {stroke} def\n");
    fprintf(prstream, "/n {newpath} def\n");
    fprintf(prstream, "/c {closepath} def\n");
    
    for (i = 0; i < number_of_colors(); i++) {
        rgb = get_rgb(i);
        if (rgb != NULL) {
            fprintf(prstream,"/Color%d {\n", i); 
            fprintf(prstream," %.4f %.4f %.4f\n", 
                               (float) rgb->red/(MAXCOLORS - 1),
                               (float) rgb->green/(MAXCOLORS - 1),
                               (float) rgb->blue/(MAXCOLORS - 1));
            fprintf(prstream,"} def\n"); 
        }
    }
    
/*
 *     fprintf(prstream, "/icolorspace {\n");
 *     fprintf(prstream, " [/Indexed /DeviceRGB %d <\n", number_of_colors());
 *     for (i = 0; i < number_of_colors(); i++) {
 *         rgb = get_rgb(i);
 *         if (rgb != NULL) {
 *             fprintf(prstream,"  %02x%02x%02x\n", 
 *                                 rgb->red, rgb->green, rgb->blue);
 *         }
 *     }
 *     fprintf(prstream, " >\n");
 *     fprintf(prstream, " ] setcolorspace\n");
 *     fprintf(prstream, "} bind def\n");
 */
   
    for (i = 0; i < number_of_patterns(); i++) {
        fprintf(prstream, "/Pat%d_bits <", i);
        for (j = 0; j < 32; j++) {
            fprintf(prstream, "%02x", pat_bits[i][j]);
        }
        fprintf(prstream, "> def\n");
        fprintf(prstream, "<<\n");
        fprintf(prstream, " /PaintType 2\n");
        fprintf(prstream, " /PatternType 1 /TilingType 1\n");
        fprintf(prstream, " /BBox[0 0 16 16]\n");
        fprintf(prstream, " /XStep 16 /YStep 16\n");
        fprintf(prstream, " /PaintProc {\n");
        fprintf(prstream, "  pop\n");
        fprintf(prstream, "  16 16 scale\n");
        fprintf(prstream, "  16 16 true [16 0 0 16 0 0] {Pat%d_bits} imagemask\n", i);
        fprintf(prstream, " }\n");
        fprintf(prstream, ">>\n");
        fprintf(prstream, "matrix\n");
        fprintf(prstream, "makepattern\n");
        fprintf(prstream, "/Pattern%d exch def\n", i);
    }
    fprintf(prstream, "/ellipsedict 8 dict def\n");
    fprintf(prstream, "ellipsedict /mtrx matrix put\n");
    fprintf(prstream, "/ellipse {\n");
    fprintf(prstream, " ellipsedict begin\n");
    fprintf(prstream, "  /endangle exch def\n");
    fprintf(prstream, "  /startangle exch def\n");
    fprintf(prstream, "  /yrad exch def\n");
    fprintf(prstream, "  /xrad exch def\n");
    fprintf(prstream, "  /y exch def\n");
    fprintf(prstream, "  /x exch def\n");
    fprintf(prstream, "  /savematrix mtrx currentmatrix def\n");
    fprintf(prstream, "  x y translate\n");
    fprintf(prstream, "  xrad yrad scale\n");
    fprintf(prstream, "  0 0 1 startangle endangle arc\n");
    fprintf(prstream, "  savematrix setmatrix\n");
    fprintf(prstream, " end\n");
    fprintf(prstream, "} def\n");

    fprintf(prstream, "%%%%EndProlog\n");

    fprintf(prstream, "%%%%BeginSetup\n");
    fprintf(prstream, "%.4f %.4f scale\n", page_scalef, page_scalef);
    if (page_orientation == PAGE_ORIENT_LANDSCAPE) {
        fprintf(prstream, "90 rotate\n");
        fprintf(prstream, "0.0 -1.0 translate\n");
    }
    fprintf(prstream, "[/Pattern /DeviceRGB] setcolorspace\n");
    fprintf(prstream, "%%%%EndSetup\n");

    if (curformat == PS_FORMAT) {
        fprintf(prstream, "%%%%Page: 1 1\n");
    }

    return GRACE_EXIT_SUCCESS;
}

void ps_setpen(void)
{
    Pen pen;
    
    pen = getpen();
    fprintf(prstream, "Color%d Pattern%d setcolor\n", pen.color, pen.pattern);
}

void ps_setlinestyle(int lines)
{
    int i;
    double lw = getlinewidth();
    
    fprintf(prstream, "[");
    if (dash_array_length[lines] > 1) {
        for (i = 0; i < dash_array_length[lines]; i++) {
            fprintf(prstream, "%.4f ",  lw*dash_array[lines][i]);
        }
    }
    fprintf(prstream, "] 0 setdash\n");
}

void ps_setlinewidth(double linew)
{
    fprintf(prstream, "%.4f setlinewidth\n", linew);
}

void ps_drawpolyline(VPoint *vps, int n, int mode)
{
    int i;
    
    fprintf(prstream, "n\n");
    fprintf(prstream, "%.4f %.4f m\n", vps[0].x, vps[0].y);
    for (i = 1; i < n; i++) {
        fprintf(prstream, "%.4f %.4f l\n", vps[i].x, vps[i].y);
    }
    if (mode == POLYLINE_CLOSED) {
        fprintf(prstream, "%.4f %.4f l\n", vps[0].x, vps[0].y);
        fprintf(prstream, "c\n");
    }
    fprintf(prstream, "s\n");
}

void ps_drawpolygon(VPoint *vps, int nc)
{
    int i;
    
    if ((getpen()).pattern == 0 || nc < 3) {
        return;
    }
    
    fprintf(prstream, "n\n");
    fprintf(prstream, "%.4f %.4f m\n", vps[0].x, vps[0].y);
    for (i = 1; i < nc; i++) {
        fprintf(prstream, "%.4f %.4f l\n", vps[i].x, vps[i].y);
    }
    fprintf(prstream, "c fill\n");
}

void ps_drawarc(VPoint vp1, VPoint vp2, int a1, int a2)
{
    VPoint vpc;
    double rx, ry;
    
    vpc.x = (vp1.x + vp2.x)/2;
    vpc.y = (vp1.y + vp2.y)/2;
    rx = fabs(vp2.x - vp1.x)/2;
    ry = fabs(vp2.y - vp1.y)/2;
    
    fprintf(prstream, "n\n");
    fprintf(prstream, "%.4f %.4f %.4f %.4f %d %d ellipse c s\n",
                       vpc.x, vpc.y, rx, ry, a1, a2);
}

void ps_fillarc(VPoint vp1, VPoint vp2, int a1, int a2)
{
    VPoint vpc;
    double rx, ry;
    
    vpc.x = (vp1.x + vp2.x)/2;
    vpc.y = (vp1.y + vp2.y)/2;
    rx = fabs(vp2.x - vp1.x)/2;
    ry = fabs(vp2.y - vp1.y)/2;
    
    fprintf(prstream, "n\n");
    fprintf(prstream, "%.4f %.4f %.4f %.4f %d %d ellipse c fill\n",
                       vpc.x, vpc.y, rx, ry, a1, a2);
}

void ps_putpixmap(VPoint vp, int width, int height, 
     char *databits, int pixmap_bpp, int bitmap_pad, int pixmap_type)
{
    int j, k;
    int cindex;
    int paddedW;
    RGB *rgb;
    unsigned char tmpbyte;

    fprintf(prstream, "gsave\n");
    fprintf(prstream, "[/DeviceRGB] setcolorspace\n");
    fprintf(prstream, "%.4f %.4f translate\n", vp.x, vp.y);
    fprintf(prstream, "%.4f %.4f scale\n", (float) width/page_scale, 
                                           (float) height/page_scale);    
    if (pixmap_bpp != 1) {
        if (pixmap_type == PIXMAP_TRANSPARENT) {
            /* TODO: mask */
        }
        fprintf(prstream, "/picstr %d string def\n", 3*width);
        fprintf(prstream, "%d %d %d\n", width, height, GRACE_BPP);
        fprintf(prstream, "[%d 0 0 %d 0 0]\n", width, -height);
        fprintf(prstream, "{currentfile picstr readhexstring pop}\n");
        fprintf(prstream, "false 3\n");
        fprintf(prstream, "colorimage\n");
        for (k = 0; k < height; k++) {
            for (j = 0; j < width; j++) {
                cindex = (databits)[k*width+j];
                rgb = get_rgb(cindex);
                fprintf(prstream, "%02x%02x%02x", rgb->red, rgb->green, rgb->blue);
            }
            fprintf(prstream, "\n");
        }
    } else {
        paddedW = PAD(width, bitmap_pad);
        if (pixmap_type == PIXMAP_OPAQUE) {
            rgb = get_rgb(getbgcolor());
            fprintf(prstream,"%.4f %.4f %.4f setcolor\n", 
                               (float) rgb->red/(MAXCOLORS - 1),
                               (float) rgb->green/(MAXCOLORS - 1),
                               (float) rgb->blue/(MAXCOLORS - 1));
            fprintf(prstream, "0 0 1 -1 rectfill\n");
        }
        rgb = get_rgb(getcolor());
        fprintf(prstream,"%.4f %.4f %.4f setcolor\n", 
                           (float) rgb->red/(MAXCOLORS - 1),
                           (float) rgb->green/(MAXCOLORS - 1),
                           (float) rgb->blue/(MAXCOLORS - 1));
        fprintf(prstream, "/picstr %d string def\n", paddedW/8);
        fprintf(prstream, "%d %d true\n", paddedW, height);
        fprintf(prstream, "[%d 0 0 %d 0 0]\n", paddedW, -height);
        fprintf(prstream, "{currentfile picstr readhexstring pop}\n");
        fprintf(prstream, "imagemask\n");
        for (k = 0; k < height; k++) {
            for (j = 0; j < paddedW/bitmap_pad; j++) {
                tmpbyte = reversebits((unsigned char) (databits)[k*paddedW/bitmap_pad + j]);
                fprintf(prstream, "%02x", tmpbyte);
            }
            fprintf(prstream, "\n");
        }
    }
    fprintf(prstream, "grestore\n");
}

void ps_puttext(VPoint start, VPoint end, double size, CompositeString *cstring)
{
    int iglyph;
    int font;
    double angle;
    double length;
    char *fontname;
    char *encscheme;
    
    size /= page_scalef;
    
    angle = (180.0/M_PI) * atan2(end.y - start.y, end.x - start.x);
    length = hypot (end.x - start.x, end.y - start.y);
        
    /* initialize pslength variable */
    fprintf(prstream, "/pslength 0 def\n");
    iglyph = 0;
    while (cstring[iglyph].s != NULL) {
        font = cstring[iglyph].font;
        if (psfont_status[font] == FALSE) {
            fontname = get_fontalias(font);
            encscheme = get_encodingscheme(font);
            fprintf(prstream, "/%s findfont\n", fontname);
            if (strcmp(encscheme, "ISOLatin1Encoding") == 0) {
                fprintf(prstream, "dup length dict begin\n");
                fprintf(prstream, " {1 index /FID ne {def} {pop pop} ifelse} forall\n");
                fprintf(prstream, " /Encoding ISOLatin1Encoding def\n");
                fprintf(prstream, " currentdict\n");
                fprintf(prstream, "end\n");
            }
            fprintf(prstream, "/Font%d exch definefont pop\n", font);
            psfont_status[font] = TRUE;
        }
        fprintf(prstream, "/Font%d findfont\n", font);
        fprintf(prstream, "%.4f scalefont\n", size*cstring[iglyph].scale);
        fprintf(prstream, "setfont\n");
        /* pop removes unneeded Y coordinate from the stack */
        fprintf(prstream, "(%s) stringwidth pop\n", escape_paren(cstring[iglyph].s));
        fprintf(prstream, "pslength add\n");
        fprintf(prstream, "/pslength exch def\n");
        iglyph++;
    }

    fprintf(prstream, "gsave\n");
    fprintf(prstream, "%.4f %.4f translate\n", start.x, start.y);
    fprintf(prstream, "%.4f rotate\n", angle);
    /*
     * Compensate for diffs between PS & T1lib 
     * (should Y be scaled the same??)
     * I use higher (.6) precision here since rounding errors will lead to
     * incorrect BB calculations
     */
    fprintf(prstream, "%.6f pslength div %.6f scale\n", 
                                    length*size, size*72.0/page_dpi_y);

    fprintf(prstream, "0.0 0.0 moveto\n");
    
    iglyph = 0;
    while (cstring[iglyph].s != NULL) {
        fprintf(prstream, "/Font%d findfont\n", cstring[iglyph].font);
        fprintf(prstream, "%.4f scalefont\n", cstring[iglyph].scale);
        fprintf(prstream, "setfont\n");
        if (cstring[iglyph].vshift != 0.0) {
            fprintf(prstream, "0.0 %.4f rmoveto\n", cstring[iglyph].vshift);
            fprintf(prstream, "(%s) show\n", escape_paren(cstring[iglyph].s));
            fprintf(prstream, "0.0 %.4f rmoveto\n", -cstring[iglyph].vshift);
        } else {
            fprintf(prstream, "(%s) show\n", escape_paren(cstring[iglyph].s));
        }
        
        if (cstring[iglyph].underline == TRUE) {
            /* TODO */
        }
        
        if (cstring[iglyph].overline == TRUE) {
            /* TODO */
        }
        
        iglyph++;
    }

    fprintf(prstream, "grestore\n");
}


void ps_leavegraphics(void)
{
    view v;
    
    if (curformat == PS_FORMAT) {
        fprintf(prstream, "showpage\n");
        fprintf(prstream, "%%%%PageTrailer\n");
    }
    fprintf(prstream, "%%%%Trailer\n");
    
    v = get_bbox(BBOX_TYPE_GLOB);
    if (page_orientation == PAGE_ORIENT_LANDSCAPE) {
        fprintf(prstream, "%%%%BoundingBox: %d %d %d %d\n", 
                                         (int) (page_scalef*(1.0 - v.yv2)) - 1,
                                         (int) (page_scalef*v.xv1) - 1,
                                         (int) (page_scalef*(1.0 - v.yv1)) + 2,
                                         (int) (page_scalef*v.xv2) + 2);
    } else {
        fprintf(prstream, "%%%%BoundingBox: %d %d %d %d\n", 
                                         (int) (page_scalef*v.xv1) - 1,
                                         (int) (page_scalef*v.yv1) - 1,
                                         (int) (page_scalef*v.xv2) + 2,
                                         (int) (page_scalef*v.yv2) + 2);
    }
    
    fprintf(prstream, "%%%%EOF\n");
}

/*
 * escape parentheses
 */
static char *escape_paren(char *s)
{
    static char *es = NULL;
    int i, elen = 0;
    
    elen = 0;
    for (i = 0; i < strlen(s); i++) {
        if (s[i] == '(' || s[i] == ')') {
            elen++;
        }
        elen++;
    }
    
    es = (char *) realloc(es, (elen + 1)*sizeof(char));
    
    elen = 0;
    for (i = 0; i < strlen(s); i++) {
        if (s[i] == '(' || s[i] == ')') {
            es[elen++] = '\\';
        }
        es[elen++] = s[i];
    }
    es[elen] = '\0';
    
    return (es);
}

int psprintinitgraphics(void)
{
    int result;
    
    result = ps_initgraphics(PS_FORMAT);
    
    if (result == GRACE_EXIT_SUCCESS) {
        curformat = PS_FORMAT;
    }
    
    return (result);
}

int epsinitgraphics(void)
{
    int result;
    
    result = ps_initgraphics(EPS_FORMAT);
    
    if (result == GRACE_EXIT_SUCCESS) {
        curformat = EPS_FORMAT;
    }
    
    return (result);
}
