#include <config.h>
#include <cmath.h>

#include <ctype.h>

#include "defines.h"
#include "globals.h"

#include "utils.h"
#include "device.h"
#include "t1fonts.h"

#include "protos.h"

static char LastEncodingFile[GR_MAXPATHLEN];
static float lastExtent;
static float lastSlant;
static int bitmap_pad;

static CompositeString *csbuf = NULL;

void (*devputpixmap) (VPoint vp, int width, int height, 
     char *databits, int pixmap_bpp, int bitmap_pad, int pixmap_type);
void (*devputtext) (VPoint start, VPoint end, double size, 
                                            CompositeString *cstring);

static int nfonts = 0;
static FontDB *FontDBtable = NULL;

int init_t1(void)
{
    int i;
    char buf[GR_MAXPATHLEN];
    FILE *fd;
    static char **Encoding = NULL;
    
    /* Set search paths: */
    sprintf(buf, "%s/fonts/type1", grace_home);
    T1_SetFileSearchPath(T1_PFAB_PATH, buf);
    T1_SetFileSearchPath(T1_AFM_PATH, buf);
    sprintf(buf, "%s/fonts/enc", grace_home);
    T1_SetFileSearchPath(T1_ENC_PATH, buf);
    
    /* Set font database: */
    sprintf(buf, "%s/fonts/FontDataBase", grace_home);
    T1_SetFontDataBase(buf);

    /* Set log-level: */
    T1_SetLogLevel(T1LOG_DEBUG);
    
#if defined(DEBUG)
#  define T1LOGFILE LOGFILE
#else
#  define T1LOGFILE NO_LOGFILE
#endif
    
    /* Initialize t1-library */
    if (T1_InitLib(T1LOGFILE|IGNORE_CONFIGFILE) == NULL) {
        return (GRACE_EXIT_FAILURE);
    }
    
    nfonts = T1_Get_no_fonts();
    if (nfonts < 1) {
        return (GRACE_EXIT_FAILURE);
    }
    
    fd = fopen(buf, "r");
    if (fd == NULL) {
        return (GRACE_EXIT_FAILURE);
    }
    
    FontDBtable = (FontDB *) malloc(nfonts*sizeof(FontDB));
    
    /* skip the first line */
    fgets(buf, GR_MAXPATHLEN - 1, fd); 
    for (i = 0; i < nfonts; i++) {
        fgets(buf, GR_MAXPATHLEN - 1, fd); 
        if (sscanf(buf, "%s %s %*s", FontDBtable[i].alias, 
                                     FontDBtable[i].encoding) != 2) {
            fclose(fd);
            return (GRACE_EXIT_FAILURE);
        }
    }
    fclose(fd);
    
    T1_SetDeviceResolutions(72.0, 72.0);
    

    Encoding = T1_LoadEncoding(T1_DEFAULT_ENCODING_FILE);
    strcpy(LastEncodingFile, T1_DEFAULT_ENCODING_FILE);
    
    if (Encoding != NULL) {
        T1_SetDefaultEncoding(Encoding);
    } else {
        errmsg("Failed loading default encoding vector!");
    }
    
    lastExtent = 1.0;
    lastSlant = T1_DEFAULT_SLANT;

    T1_AASetBitsPerPixel(GRACE_BPP);
    
    bitmap_pad = T1_GetBitmapPad();
    
    return (GRACE_EXIT_SUCCESS);
}

void update_t1(void)
{
    int i;
    
    float Slant = T1_DEFAULT_SLANT, Extent;
    char EncodingFile[GR_MAXPATHLEN] = T1_DEFAULT_ENCODING_FILE;
    
    static char **Encoding = NULL;
    
    if (strcmp(EncodingFile, LastEncodingFile)) {
      	/* Delete all size dependent data */
      	for (i = 0; i < T1_Get_no_fonts(); i++) {
  	    T1_DeleteAllSizes(i);
  	    T1_LoadFont(i);
      	}
      	Encoding = T1_LoadEncoding(EncodingFile);
      	for (i = 0; i < T1_Get_no_fonts(); i++) {
  	    T1_ReencodeFont(i, Encoding);
      	}
      	strcpy(LastEncodingFile, EncodingFile);
    }
    if (Slant != lastSlant) {
      	/* Delete all size dependent data */
      	for (i = 0; i < T1_Get_no_fonts(); i++) {
  	    T1_DeleteAllSizes(i);
  	    T1_LoadFont(i);
      	}
      	for (i = 0; i < T1_Get_no_fonts(); i++) {
  	    T1_SlantFont(i, Slant);
      	}
      	lastSlant = Slant;
    }

    Extent = page_dpi_x/page_dpi_y;
    if (Extent != lastExtent) {
      	/* Delete all size dependent data */
      	for (i = 0; i < T1_Get_no_fonts(); i++) {
  	    T1_DeleteAllSizes(i);
  	    T1_LoadFont(i);
      	}
      	for (i = 0; i < T1_Get_no_fonts(); i++) {
  	    T1_ExtendFont(i, Extent);
      	}
      	lastExtent=Extent;
    }    
}

int number_of_fonts(void)
{
    return (nfonts);
}

char *get_fontfilename(int font)
{
    return (T1_GetFontFileName(font));
}

char *get_fontname(int font)
{
    return (T1_GetFontName(font));
}

char *get_fontalias(int font)
{
    return (FontDBtable[font].alias);
}

char *get_encodingscheme(int font)
{
    return (T1_GetEncodingScheme(font));
}

GLYPH *GetGlyphString(int FontID, double Size, double Angle, int modflag, 
                                                            char *theString)
{
    int len, i, j, k, l, m, none_found;
    
/*
 *     int Kerning = 0;
 */
    long Space = 0;
    int LigDetect = 0;
    
    GLYPH *glyph;

    char *ligtheString = '\0';
    char *succs, *ligs;
    char buf_char;

    static int aacolors[T1_AALEVELS];
    unsigned int fg, bg;
    static unsigned long last_bg = 0, last_fg = 0;

    RGB fg_rgb, bg_rgb, delta_rgb, *prgb;
    CMap_entry cmap;
    
    Device_entry dev;

    if (strcmp(theString, "") == 0) {
        return NULL;
    }

    if (Size <= 0.0) {
        errmsg("t1lib: Size must be positive!");
        return NULL;
    }

    /* Now comes the ligatur handling */
    len = strlen(theString);
    ligtheString = (char *) malloc((len + 1)*sizeof(char));
    if (LigDetect){
      	for (j = 0, m = 0; j < len; j++, m++) { /* Loop through the characters */
  	    if ((k = T1_QueryLigs(FontID, theString[j], &succs, &ligs)) > 0) {
  	      	buf_char = theString[j];
  	      	while (k > 0){
  	    	    none_found = 1;
  	    	    for (l = 0; l < k; l++) { /* Loop through the ligatures */
  	    	      	if (succs[l] == theString[j + 1]) {
  	    	    	    buf_char = ligs[l];
  	    	    	    j++;
  	    	    	    none_found = 0;
  	    	    	    break;
  	    	      	}
  	    	    }
  	    	    if (none_found)
  	    	        break;
  	    	    k = T1_QueryLigs(FontID, buf_char, &succs, &ligs);
  	      	}
  	      	ligtheString[m] = buf_char;
  	    } else { /* There are no ligatures */
  	        ligtheString[m] = theString[j];
  	    }
      	}
      	ligtheString[m] = 0;
    }
    else {
        strcpy(ligtheString, theString);
    }

    dev = get_curdevice_props();
    if (dev.fontaa == TRUE) {
    	fg = getcolor();
    	bg = getbgcolor();

    	aacolors[0] = bg;
    	aacolors[T1_AALEVELS - 1] = fg;

    	if ((fg != last_fg) || (bg != last_bg)) {
    	    /* Get RGB values for fore- and background */
    	    prgb = get_rgb(fg);
    	    if (prgb == NULL) {
    		return NULL;
    	    }
    	    fg_rgb = *prgb;
 
    	    prgb = get_rgb(bg);
    	    if (prgb == NULL) {
    		return NULL;
    	    }
    	    bg_rgb = *prgb;
 
    	    delta_rgb.red   = (fg_rgb.red   - bg_rgb.red)   / (T1_AALEVELS - 1);
    	    delta_rgb.green = (fg_rgb.green - bg_rgb.green) / (T1_AALEVELS - 1);
    	    delta_rgb.blue  = (fg_rgb.blue  - bg_rgb.blue) / (T1_AALEVELS - 1);
 
    	    for (i = 1; i < T1_AALEVELS - 1; i++) {
    		cmap.rgb.red   = bg_rgb.red + i*delta_rgb.red;
    		cmap.rgb.green = bg_rgb.green + i*delta_rgb.green;
    		cmap.rgb.blue  = bg_rgb.blue + i*delta_rgb.blue;
    		cmap.cname = "";
    		cmap.ctype = COLOR_AUX;
    		aacolors[i] = add_color(cmap);
    	    }
 
    	    last_fg = fg;
    	    last_bg = bg;
    	}
 
    	/* Set the colors for Anti-Aliasing */
    	T1_AASetGrayValues(aacolors[0],
    			   aacolors[1],
    			   aacolors[2],
    			   aacolors[3],
    			   aacolors[4]);

    	glyph = T1_AASetString(FontID, ligtheString, 0,
    				   Space, modflag, (float) Size, (float) Angle);
    } else {
    	glyph = T1_SetString(FontID, ligtheString, 0,
    				   Space, modflag, (float) Size, (float) Angle);
    }
 
    free(ligtheString);
 
    if ((glyph == NULL) ||
  	(glyph->metrics.ascent - glyph->metrics.descent == 0) ||
  	(glyph->metrics.rightSideBearing - glyph->metrics.leftSideBearing == 0)) {
/*
 *         errmsg("t1lib: Invalid Raster-Parameter(s)!");
 */
        return NULL;
    }

    return glyph;
}

void FreeCompositeString(CompositeString *cs)
{
    int i = 0;
    
    while (cs[i].s != NULL) {
	free (cs[i].s);
	i++;
    }
    free (cs);
}

CompositeString *String2Composite(char *string)
{

    char ss[MAX_STRING_LENGTH];
    int slen;
    int nss;
    int i, isub;
    
    int upperset = FALSE;
    int underline = FALSE, overline = FALSE;
    int new_underline = underline, new_overline = overline;
    double vshift = 0.0;
    double new_vshift = vshift; 
    double scale = 1.0;
    double new_scale = scale;
    int font = draw_props.font;
    int new_font = font;

    
    slen = strlen(string);
    
    if ((slen == 0) || slen > (MAX_STRING_LENGTH - 1)) {
        return NULL;
    }
     
    if (csbuf != NULL) {
        FreeCompositeString(csbuf);
	csbuf = NULL;
    }
    
    nss = 0;
    
    isub = 0;
    ss[isub] = 0;
    
    for (i = 0; i <= slen; i++) {
/*
 * 	if (string[i] < 32) {
 * 	    continue;
 * 	}
 */
	if (string[i] == '\\' && isdigit(string[i + 1])) {
	    new_font = string[i + 1] - '0';
	    i++;
	    continue;
	} else if (string[i] == '\\' && string[i + 1] == '\\') {
	    if (!upperset) {	/* special case */
	    	i++;
	    }
	} else if (string[i] == '\\' && isoneof(string[i + 1], "cCsSNBuUoO+-")) {
	    switch (string[i + 1]) {
	    case 's':
		new_scale *= SSCRIPT_SCALE;
		new_vshift -= 0.4*scale;
		i++;
		break;
	    case 'S':
		new_scale *= SSCRIPT_SCALE;
		new_vshift += 0.6*scale;
		i++;
		break;
	    case 'N':
		new_scale = 1.0;
		new_vshift = 0.0;
		i++;
		break;
	    case 'B':
		new_font = draw_props.font;
		i++;
		break;
	    case 'c':
	        upperset = TRUE;
		i++;
		break;
	    case 'C':
	        upperset = FALSE;
		i++;
		break;
	    case 'u':
		new_underline = TRUE;
		i++;
		break;
	    case 'U':
		new_underline = FALSE;
		i++;
		break;
	    case 'o':
		new_overline = TRUE;
		i++;
		break;
	    case 'O':
		new_overline = FALSE;
		i++;
		break;
	    case '-':
		new_scale /= ENLARGE_SCALE;
		i++;
		break;
	    case '+':
		new_scale *= ENLARGE_SCALE;
		i++;
		break;
	    }
	    continue;
	}
	if ((new_font  != font          ) ||
	    (new_scale != scale         ) ||
	    (new_underline != underline ) ||
	    (new_overline != overline   ) ||
	    (string[i] == 0             )) {
	    
	    
            if (isub != 0) {	/* non-empty substring */
                ss[isub] = 0;
	        isub = 0;
	
	        csbuf = (CompositeString *) realloc(csbuf, (nss + 1)*sizeof(CompositeString));
	        csbuf[nss].font = font;
	        csbuf[nss].scale = scale;
	        csbuf[nss].vshift = vshift;
	        csbuf[nss].underline = underline;
	        csbuf[nss].overline = overline;
	        csbuf[nss].s = (char *) malloc(strlen(ss) + 1);
	        strcpy(csbuf[nss].s, ss);
	
                nss++;
            }
	    
	    font = new_font;
	    scale = new_scale;
	    vshift = new_vshift;
	    underline = new_underline;
	    overline = new_overline;
	} 
	ss[isub] = (string[i] + (upperset*0x80)) & 0xff;
	isub++;
    }
    csbuf = (CompositeString *) realloc(csbuf, (nss + 1)*sizeof(CompositeString));
    csbuf[nss].s = NULL;
    
    return (csbuf);
}

/*
 * Convenience wrapper for T1_ConcatGlyphs()
 */
GLYPH *CatGlyphs(GLYPH *dest_glyph, GLYPH *src_glyph, int x_off, int y_off)
{
    GLYPH *buf_glyph;
    
    if (src_glyph == NULL) {
        return (dest_glyph);
    }
    
    if (dest_glyph != NULL) {
        buf_glyph = T1_ConcatGlyphs(dest_glyph, src_glyph, x_off, y_off);
        T1_FreeGlyph(dest_glyph);
        dest_glyph = T1_CopyGlyph(buf_glyph);
    } else {
        dest_glyph = T1_CopyGlyph(src_glyph);
    }
    
    return (dest_glyph);
}

#define T1_CONCAT_FEATURE

void WriteString(VPoint vp, int rot, int just, char *theString)
{    
    VPoint vptmp;
 
    int hjust, vjust, just_type;
    float hfudge, vfudge;
    
    double page_ipv, page_dpv_x, page_dpv_y;
 
    /* Variables for raster parameters */
    double Size, Angle = 0.0;
    int FontID;
    int modflag;

    int iglyph;
    GLYPH *glyph;
    GLYPH *CSglyph = NULL;
 
    CompositeString *cstring;
 
    double scale_factor;
    
    int pheight, pwidth;
    
    float v_off;
    float v_off_buf, v_off_old = 0.0;
    int x_off = 0, y_off = 0;
    
    int baseline_start_x, baseline_start_y, baseline_end_x, baseline_end_y;
    int bbox_left_x, bbox_right_x, bbox_lower_y, bbox_upper_y;
    int pinpoint_x, pinpoint_y, justpoint_x, justpoint_y;

    int xshift, yshift;
    
    VPoint vp_baseline_start, vp_baseline_end;
    
    Device_entry dev;
 
    if (theString == NULL || strlen(theString) == 0) {
	return;
    }
    
    dev = get_curdevice_props();
    
    cstring = String2Composite(theString);
    
    /* inches per 1 unit of viewport */
    page_ipv = MIN2(page_width_in, page_height_in);

    /* dots per 1 unit of viewport */
    page_dpv_x = page_ipv*page_dpi_x;
    page_dpv_y = page_ipv*page_dpi_y;

    scale_factor = page_dpv_y * MAGIC_FONT_SCALE * getcharsize();
    
    iglyph = 0;
    while (cstring[iglyph].s != NULL) {
        Size = scale_factor * cstring[iglyph].scale;
  	Angle = (double) rot;
  	FontID = cstring[iglyph].font;
        modflag = T1_UNDERLINE * cstring[iglyph].underline |
                  T1_OVERLINE  * cstring[iglyph].overline;
	glyph = GetGlyphString(FontID, Size, Angle, modflag, cstring[iglyph].s);
        v_off = scale_factor * cstring[iglyph].vshift;
        v_off_buf = v_off;
        v_off -= v_off_old;
        v_off_old = v_off_buf;
        x_off = - (int) (v_off*sin(M_PI/180.0*Angle));
        y_off = + (int) (v_off*cos(M_PI/180.0*Angle));
        CSglyph = CatGlyphs(CSglyph, glyph, x_off, y_off);
	iglyph++;
    }
    if (CSglyph == NULL) {
        return;
    }
    
    /* we'll need this later to find the endpoint of the baseline of the
       whole composite string */
    if (dev.devfonts == FONTSRC_DEVICE) {
        x_off = - (int) (v_off_old*sin(M_PI/180.0*Angle));
        y_off = + (int) (v_off_old*cos(M_PI/180.0*Angle));
    }
    
    pinpoint_x = CSglyph->metrics.leftSideBearing;
    pinpoint_y = CSglyph->metrics.ascent;
    
    baseline_start_x = 0;
    baseline_start_y = 0;
    baseline_end_x = CSglyph->metrics.advanceX;
    baseline_end_y = CSglyph->metrics.advanceY;

    bbox_left_x =  MIN3(baseline_start_x, CSglyph->metrics.leftSideBearing,  baseline_end_x);
    bbox_right_x = MAX3(baseline_start_x, CSglyph->metrics.rightSideBearing, baseline_end_x);
    bbox_lower_y = MIN3(baseline_start_y, CSglyph->metrics.descent, baseline_end_y);
    bbox_upper_y = MAX3(baseline_start_y, CSglyph->metrics.ascent,  baseline_end_y);
    
    hjust = just & 03;
    switch (hjust) {
    case JUST_LEFT:
        hfudge = 0.0;
        break;
    case JUST_RIGHT:
        hfudge = 1.0;
        break;
    case JUST_CENTER:
        hfudge = 0.5;
        break;
    default:
        errmsg("Wrong justification type of string");
        return;
    }

    vjust = just & 014;
    switch (vjust) {
    case JUST_BOTTOM:
        vfudge = 0.0;
        break;
    case JUST_TOP:
        vfudge = 1.0;
        break;
    case JUST_MIDDLE:
        vfudge = 0.5;
        break;
    default:
        errmsg("Wrong justification type of string");
        return;
    }
 
    just_type = just & 020;
    switch (just_type) {
    case JUST_OBJECT:
        justpoint_x = (int) (baseline_start_x + 
                                hfudge*(baseline_end_x - baseline_start_x));
        justpoint_y = (int) (baseline_start_y + 
                                hfudge*(baseline_end_y - baseline_start_y));
        break;
    case JUST_BBOX:
        justpoint_x = (int) (bbox_left_x + 
                                hfudge*(bbox_right_x - bbox_left_x));
        justpoint_y = (int) (bbox_lower_y + 
                                vfudge*(bbox_upper_y - bbox_lower_y));
        break;
    default:
        errmsg("Wrong justification type of string");
        return;
    }
 
    pheight = CSglyph->metrics.ascent - CSglyph->metrics.descent;
    pwidth = CSglyph->metrics.rightSideBearing - CSglyph->metrics.leftSideBearing;

    xshift = pinpoint_x - justpoint_x;
    yshift = pinpoint_y - justpoint_y;
    
    vptmp.x = vp.x + (double) xshift/page_dpv_x;
    vptmp.y = vp.y + (double) yshift/page_dpv_y;

    if (get_draw_mode() == TRUE) {
        if (dev.devfonts == FONTSRC_BITMAP) {
            (*devputpixmap) (vptmp, pwidth, pheight, CSglyph->bits, 
                                CSglyph->bpp, bitmap_pad, PIXMAP_TRANSPARENT);
        } else {
            if (devputtext == NULL) {
                errmsg("Device has no fonts built-in");
            } else {
                vp_baseline_start.x = vptmp.x + 
                    (double) (baseline_start_x - bbox_left_x)/page_dpv_x;
                vp_baseline_start.y = vptmp.y +
                    (double) (baseline_start_y - bbox_upper_y)/page_dpv_y;
                vp_baseline_end.x   = vptmp.x + 
                    (double) (baseline_end_x - bbox_left_x - x_off)/page_dpv_x;
                vp_baseline_end.y   = vptmp.y +
                    (double) (baseline_end_y - bbox_upper_y - y_off)/page_dpv_y;
                (*devputtext) (vp_baseline_start, vp_baseline_end, 
                                                scale_factor, cstring);
            }
        }
    }

    T1_FreeGlyph(CSglyph);
    
    update_bboxes(vptmp);
    vptmp.x += (double) pwidth/page_dpv_x;
    vptmp.y -= (double) pheight/page_dpv_y;
    update_bboxes(vptmp);
}
