/* 
 * Draw axis bars, axis labels, ticks and tick labels
 */

#include <config.h>
#include <cmath.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "globals.h"
#include "utils.h"
#include "draw.h"
#include "t1fonts.h"
#include "protos.h"

int is_xaxis(int axis)
{
    return ((axis % 2 == 0));
}

void drawgrid(int gno)
{
    int caxis;
    tickmarks t;
    tickprops tprops;
    int ttype;
    world w;
    view v;
    double vbase_start, vbase_stop;
    double wtpos, vtpos;
    VPoint vp_grid_start, vp_grid_stop;
    VVector ort_para, ort_perp;
    double wc_start, wc_stop; /* world coordinates */
    int ittype_loop, itick;
    
    double (*coord_conv) ();
    
    setclipping(TRUE);
    
    /* TODO: add Pen to ticks and remove the following */
    setpattern(1);
    
    get_graph_view(gno, &v);
    get_graph_world(gno, &w);
    
    for (caxis = 0; caxis < MAXAXES; caxis++) {
	get_graph_tickmarks(gno, &t, caxis);
	if (t.active != TRUE) {
            continue;
        }
	if (is_xaxis(caxis)) { /* an X-axis */
	    ort_para.x = 1.0;
	    ort_para.y = 0.0;
	    ort_perp.x = 0.0;
	    ort_perp.y = 1.0;
	    
	    vbase_start = v.yv1;
	    vbase_stop = v.yv2;
	       
	    wc_start = w.xg1;
	    wc_stop = w.xg2;
	    
	    coord_conv = xconv;
	    
	} else {              /* a Y-axis */
	    ort_para.x = 0.0;
	    ort_para.y = 1.0;
	    ort_perp.x = 1.0;
	    ort_perp.y = 0.0;
	    
	    vbase_start = v.xv1;
	    vbase_stop = v.xv2;
	    
	    wc_start = w.yg1;
	    wc_stop = w.yg2;
	    
	    coord_conv = yconv;
	}
		
	for (ittype_loop = 0; ittype_loop < 2; ittype_loop++) {
	    if (ittype_loop == 0) { /* minor ticks */
                ttype = TICK_TYPE_MINOR;
                tprops = t.mprops;
	    } else {	  /* major ticks */
                ttype = TICK_TYPE_MAJOR;
                tprops = t.props;
	    }
            if (tprops.gridflag == 0) {
	        continue;
	    }

            setcolor(tprops.color);
	    setlinewidth(tprops.linew);
	    setlinestyle(tprops.lines);
	    
	    for (itick = 0; itick < t.nticks; itick++) {
	    	if (t.tloc[itick].type != ttype) {
	    	    continue;
	    	}
                
                wtpos = t.tloc[itick].wtpos;
	    	
	    	if ((wtpos < wc_start) || (wtpos > wc_stop)) {
	    	    continue;
	    	}
	    	vtpos = coord_conv(wtpos);
	    	
		vp_grid_start.x = vtpos*ort_para.x + vbase_start*ort_perp.x;
		vp_grid_start.y = vtpos*ort_para.y + vbase_start*ort_perp.y;
		vp_grid_stop.x  = vtpos*ort_para.x + vbase_stop*ort_perp.x;
		vp_grid_stop.y  = vtpos*ort_para.y + vbase_stop*ort_perp.y;
		
		DrawLine(vp_grid_start, vp_grid_stop);
	    }
	}
    }
}

void drawaxes(int gno)
{
    int caxis;
    tickmarks t;
    tickprops tprops;
    world w;
    view v;
    double vbase1, vbase2, vbase1_start, vbase1_stop, vbase2_start, vbase2_stop;
    double vbase_tlabel, vbase_tlabel1, vbase_tlabel2;
    double tsize, tlsize, wtpos, vtpos, tick_value, tl_offset;
    VPoint vp1_start, vp1_stop, vp2_start, vp2_stop;
    VPoint vp_tick1_start, vp_tick1_stop, vp_tick2_start, vp_tick2_stop;
    VPoint vp_tlabel1, vp_tlabel2, vp_label, vp_label_offset;
    VVector ort_para, ort_perp;
    double wc_start, wc_stop, wc_start_labels, wc_stop_labels; /* world
                                                                coordinates */
    int ittype_loop, itick, itcur;
    int ttype;
    char tlabel[MAX_STRING_LENGTH];
    int tlabel1_just, tlabel2_just;
    int langle;
    
    double (*coord_conv) ();
    
    tl_offset = 0.0125; /* distance between major ticks and tick labels */
    
    setclipping(FALSE);

    /* TODO: add Pen to ticks and remove the following */
    setpattern(1);
    
    get_graph_view(gno, &v);
    get_graph_world(gno, &w);
    
    for (caxis = 0; caxis < MAXAXES; caxis++) {
	get_graph_tickmarks(gno, &t, caxis);
	if (t.active != TRUE) {
            continue;
        }
        if (is_xaxis(caxis)) { /* an X-axis */
	    ort_para.x = 1.0;
	    ort_para.y = 0.0;
	    ort_perp.x = 0.0;
	    ort_perp.y = 1.0;
	    
	    vbase1 = v.yv1 - t.offsx;
	    vbase2 = v.yv2 + t.offsy;
	    
	    vp1_start.x = v.xv1;
	    vp1_start.y = vbase1;
	    vp1_stop.x  = v.xv2;
	    vp1_stop.y  = vbase1;
	    vp2_start.x = v.xv1;
	    vp2_start.y = vbase2;
	    vp2_stop.x  = v.xv2;
	    vp2_stop.y  = vbase2;
	    
	    wc_start = w.xg1;
	    wc_stop = w.xg2;
	    
	    coord_conv = xconv;
	    
	    tlabel1_just = JUST_CENTER|JUST_TOP|JUST_BBOX;
	    tlabel2_just = JUST_CENTER|JUST_BOTTOM|JUST_BBOX;
	    
	    switch (t.label_layout) {
	    case LAYOUT_PARALLEL:
	    	langle =  0;
	    	break;
	    case LAYOUT_PERPENDICULAR:
	    	langle = 90;
	    	break;
	    default:
	    	errmsg("Internal error in drawaxes()");
	    	return;
	    }
	    vp_label_offset.x =  0.00;
	    vp_label_offset.y = -0.08;
	} else {              /* a Y-axis */
	    ort_para.x = 0.0;
	    ort_para.y = 1.0;
	    ort_perp.x = 1.0;
	    ort_perp.y = 0.0;
	    
	    vbase1 = v.xv1 - t.offsx;
	    vbase2 = v.xv2 + t.offsy;
	    
	    vp1_start.x = vbase1;
	    vp1_start.y = v.yv1;
	    vp1_stop.x  = vbase1;
	    vp1_stop.y  = v.yv2;
	    vp2_start.x = vbase2;
	    vp2_start.y = v.yv1;
	    vp2_stop.x  = vbase2;
	    vp2_stop.y  = v.yv2;
	    
	    wc_start = w.yg1;
	    wc_stop = w.yg2;
	    
	    coord_conv = yconv;
	    
	    tlabel1_just = JUST_RIGHT|JUST_MIDDLE|JUST_BBOX;
	    tlabel2_just = JUST_LEFT|JUST_MIDDLE|JUST_BBOX;
	
	    switch (t.label_layout) {
	    case LAYOUT_PARALLEL:
	    	langle = 90;
	    	break;
	    case LAYOUT_PERPENDICULAR:
	    	langle =  0;
	    	break;
	    default:
	    	errmsg("Internal error in drawaxes()");
	    	return;
	    }
	    vp_label_offset.x = -0.08;
	    vp_label_offset.y =  0.00;
	}
	
	
	/* Begin axis bar stuff */
	if (t.t_drawbar) {
	    setcolor(t.t_drawbarcolor);
	    setlinewidth(t.t_drawbarlinew);
	    setlinestyle(t.t_drawbarlines);
	    if (t.t_op == PLACE_LEFT || t.t_op == PLACE_BOTTOM || t.t_op == PLACE_BOTH) {
	    	DrawLine(vp1_start, vp1_stop);
	    }
	    if (t.t_op == PLACE_RIGHT || t.t_op == PLACE_TOP || t.t_op == PLACE_BOTH) {
	    	DrawLine(vp2_start, vp2_stop);
	    }
	}
	/* End axis bar stuff*/
	
	/* Begin axis tick & tick labels stuff */
	if (t.tl_starttype == TYPE_SPEC) {
	    wc_start_labels = t.tl_start;
        } else {
	    wc_start_labels = wc_start;
	}
	
	if (t.tl_stoptype == TYPE_SPEC) {
	    wc_stop_labels = t.tl_stop;
        } else {
	    wc_stop_labels = wc_stop;
	}
		
	tlsize = 0.02 * t.tl_charsize;
	
	for (ittype_loop = 0; ittype_loop < 2; ittype_loop++) {
	    if (ittype_loop == 0) { /* minor ticks */
                ttype = TICK_TYPE_MINOR;
                tprops = t.mprops;
	    } else {	  /* major ticks */
                ttype = TICK_TYPE_MAJOR;
                tprops = t.props;
	    }
	    tsize = 0.02 * tprops.size;
            
	    switch (t.t_inout) {
	    case TICKS_IN:
	    	vbase1_start = vbase1;
	    	vbase1_stop  = vbase1 + tsize;
	    	vbase2_start = vbase2;
	    	vbase2_stop  = vbase2 - tsize;
	    	vbase_tlabel1 = vbase1 - tl_offset;
	    	vbase_tlabel2 = vbase2 + tl_offset;
	    	break;
	    case TICKS_OUT:
	    	vbase1_start = vbase1;
	    	vbase1_stop  = vbase1 - tsize;
	    	vbase2_start = vbase2;
	    	vbase2_stop  = vbase2 + tsize;
	    	vbase_tlabel1 = vbase1 - tsize - tl_offset;
	    	vbase_tlabel2 = vbase2 + tsize + tl_offset;
	    	break;
	    case TICKS_BOTH:
	    	vbase1_start = vbase1 - tsize;
	    	vbase1_stop  = vbase1 + tsize;
	    	vbase2_start = vbase2 + tsize;
	    	vbase2_stop  = vbase2 - tsize;
	    	vbase_tlabel1 = vbase1 - tsize - tl_offset;
	    	vbase_tlabel2 = vbase2 + tsize + tl_offset;
	    	break;
	    default:
	    	errmsg("Internal error in drawaxes()");
	    	return;
	    }
	
	    itcur = 0;
            for (itick = 0; itick < t.nticks; itick++) {
	    	if (t.tloc[itick].type != ttype) {
	    	    continue;
	    	}
                
                wtpos = t.tloc[itick].wtpos;
	
	    	if ((wtpos < wc_start) || (wtpos > wc_stop)) {
	    	    continue;
	    	}
	    	
	    	if (t.tl_type == TYPE_SPEC) {
	    	    if (t.t_speclab[itick].s != NULL) {
                        strcpy(tlabel, t.t_speclab[itick].s);
                    } else {
                        tlabel[0] = '\0';
                    }
	    	} else {
	    	    switch (t.tl_sign) {
	    	    case SIGN_NORMAL:
	    		tick_value = wtpos;
	    		break;
	    	    case SIGN_ABSOLUTE:
	    		tick_value = fabs(wtpos);
	    		break;
	    	    case SIGN_NEGATE:
	    		tick_value = -wtpos;
	    		break;
	    	    default:
	    		errmsg("Internal error in drawaxes()");
	    		return;
	    	    }
	    	    strcpy(tlabel, create_fstring(t.tl_format, t.tl_prec, 
                            tick_value, LFORMAT_TYPE_EXTENDED));
	    	}
	    	if (t.tl_prestr[0]) {
	    	    char tmpbuf[MAX_STRING_LENGTH];
	
	    	    strcpy(tmpbuf, t.tl_prestr);
	    	    strcat(tmpbuf, tlabel);
	    	    strcpy(tlabel, tmpbuf);
	    	}
	    	if (t.tl_appstr[0]) {
	    	    strcat(tlabel, t.tl_appstr);
	    	}
	    	
	    	vtpos = coord_conv(wtpos);
	    	if (t.t_flag) {
	    	
                    setcolor(tprops.color);
                    setlinewidth(tprops.linew);
	            setlinestyle(tprops.lines);
	    	    
	    	    if (t.t_op == PLACE_LEFT ||
	    		t.t_op == PLACE_BOTTOM ||
	    		t.t_op == PLACE_BOTH) {
	    		vp_tick1_start.x = vtpos*ort_para.x + vbase1_start*ort_perp.x;
	    		vp_tick1_start.y = vtpos*ort_para.y + vbase1_start*ort_perp.y;
	    		vp_tick1_stop.x  = vtpos*ort_para.x + vbase1_stop*ort_perp.x;
	    		vp_tick1_stop.y  = vtpos*ort_para.y + vbase1_stop*ort_perp.y;
	    		DrawLine(vp_tick1_start, vp_tick1_stop);
	    	    }
	    	    if (t.t_op == PLACE_RIGHT ||
	    		t.t_op == PLACE_TOP ||
	    		t.t_op == PLACE_BOTH) {
	    		vp_tick2_start.x = vtpos*ort_para.x + vbase2_start*ort_perp.x;
	    		vp_tick2_start.y = vtpos*ort_para.y + vbase2_start*ort_perp.y;
	    		vp_tick2_stop.x  = vtpos*ort_para.x + vbase2_stop*ort_perp.x;
	    		vp_tick2_stop.y  = vtpos*ort_para.y + vbase2_stop*ort_perp.y;
	    		DrawLine(vp_tick2_start, vp_tick2_stop);
	    	    }
	    	}
	    	
	    	if (t.tl_flag && 
	    	    (ttype == TICK_TYPE_MAJOR) && itcur % (t.tl_skip + 1) == 0) {

	    	    if ((wtpos < wc_start_labels) || (wtpos > wc_stop_labels)) {
	    	        continue;
	    	    }
	    	    
	    	    setcolor(t.tl_color);
	    	    setfont(t.tl_font);
	    	    setcharsize(t.tl_charsize);
	    	    
	    	    if (t.tl_op == PLACE_LEFT || 
	    		t.tl_op == PLACE_BOTTOM || 
	    		t.tl_op == PLACE_BOTH) {
	    		vbase_tlabel = vbase_tlabel1 - (tl_offset + tlsize)*
	    					   (itcur % (t.tl_staggered + 1));
	    		vp_tlabel1.x = vtpos*ort_para.x + vbase_tlabel*ort_perp.x;
	    		vp_tlabel1.y = vtpos*ort_para.y + vbase_tlabel*ort_perp.y;
	    		WriteString(vp_tlabel1, t.tl_angle, tlabel1_just, tlabel);
	    	    }
	    	    if (t.tl_op == PLACE_RIGHT ||
	    		t.tl_op == PLACE_TOP || 
	    		t.tl_op == PLACE_BOTH) {
	    		vbase_tlabel = vbase_tlabel2 + (tl_offset + tlsize)*
	    					   (itcur % (t.tl_staggered + 1));
	    		vp_tlabel2.x = vtpos*ort_para.x + vbase_tlabel*ort_perp.x;
	    		vp_tlabel2.y = vtpos*ort_para.y + vbase_tlabel*ort_perp.y;
	    		WriteString(vp_tlabel2, t.tl_angle, tlabel2_just, tlabel);
	    	    }
	    	}
                itcur++;
	    }
	
	}
	/* End axis ticks stuff */
	
	/* Begin axis label stuff */
	if (t.label.s[0]) {
	    
	    if (t.label_place == TYPE_SPEC) {
	        vp_label_offset.x = t.label.x;
		vp_label_offset.y = t.label.y;
	    } else {
		t.label.x = vp_label_offset.x;
		t.label.y = vp_label_offset.y;
	    }
	    vp_label.x = (vp1_start.x + vp1_stop.x)/2 + vp_label_offset.x;
	    vp_label.y = (vp1_start.y + vp1_stop.y)/2 + vp_label_offset.y;
	    
	    setcharsize(t.label.charsize);
	    setfont(t.label.font);
	    setcolor(t.label.color);
	    WriteString(vp_label, langle, JUST_CENTER|JUST_MIDDLE|JUST_BBOX, t.label.s);
	}
	/* End axis label stuff */
    }
}

void calculate_tickgrid(int gno)
{
    extern double fscale(double wc, int scale);
    extern double ifscale(double vc, int scale);
    
    
    int caxis;
    int itick, imtick, itmaj;
    int nmajor;
    double swc_start, swc_stop;
    int scale;
    double wtmaj;
    world w;
    tickmarks t;
    
reenter:
    get_graph_world(gno, &w);
    
    for (caxis = 0; caxis < MAXAXES; caxis++) {
	get_graph_tickmarks(gno, &t, caxis);

	if (t.active != TRUE) {
            continue;
        }
        
        if (t.t_type == TYPE_SPEC) {
            continue;
        }
        
        
        if (is_xaxis(caxis)) {
            scale = g[gno].xscale;
            if (scale == SCALE_LOG) {
                swc_start = fscale(w.xg1, scale);
                swc_stop  = fscale(w.xg2, scale);
            } else {
                swc_start = w.xg1;
                swc_stop  = w.xg2;
            }
        } else {
            scale = g[gno].yscale;
            if (scale == SCALE_LOG) {
                swc_start = fscale(w.yg1, scale);
                swc_stop  = fscale(w.yg2, scale);
            } else {
                swc_start = w.yg1;
                swc_stop  = w.yg2;
            }
        }
        if (scale == SCALE_LOG) {
            nmajor = (int) rint ((swc_stop - swc_start) / fscale(t.tmajor, scale) + 1);
        } else {
            nmajor = (int) rint ((swc_stop - swc_start) / t.tmajor + 1);
        }
        t.nticks = (nmajor - 1)*(t.nminor + 1) + 1;
        
        if (t.nticks > MAX_TICKS) {
	    errmsg("Too many ticks ( > MAX_TICKS ), autoticking");
	    autotick_axis(gno, caxis);
            goto reenter;
	}

/*
 *         if (t.nticks > MAX_TICKS) {
 *             t.nticks = MAX_TICKS;
 *         }
 */
        
        itick = 0;
        itmaj = 0;
        while (itick < t.nticks) {
            if (scale == SCALE_LOG) {
                wtmaj = ifscale(swc_start + itmaj*fscale(t.tmajor, scale), scale);
            } else {
                wtmaj = swc_start + itmaj*t.tmajor;
            }
            t.tloc[itick].wtpos = wtmaj;
            t.tloc[itick].type = TICK_TYPE_MAJOR;
            itick++;
            for (imtick = 0; imtick < t.nminor && itick < t.nticks; imtick++) {
                if (scale == SCALE_LOG) {
                    t.tloc[itick].wtpos = wtmaj * (imtick + 2);
                } else {
                    t.tloc[itick].wtpos = wtmaj + (imtick + 1)*t.tmajor/(t.nminor + 1);
                }
                t.tloc[itick].type = TICK_TYPE_MINOR;
                itick++;
            }
            itmaj++;
        }
        set_graph_tickmarks(gno, &t, caxis);
    }
}
