/* $Id: graphu1.c,v 1.4 1995/07/01 04:53:30 pturner Exp pturner $
 *
 * utilities for graphs
 *
 */

#include <config.h>
#include <cmath.h>

#include <stdio.h>

#include "globals.h"
#include "draw.h"
#include "protos.h"

static void default_axis(int gno, int method, int axis);
static void auto_ticks(int gno, int axis, double *gmin, double *gmax);

/*
 * Count the number of active sets in graph gno
 */
int nactive(int gno)
{
    int i, cnt = 0;
    for (i = 0; i < g[gno].maxplot; i++) {
	if (isactive_set(gno, i)) {
	    cnt++;
	}
    }
    return cnt;
}

int isxreversed(int gno)
{
    return (g[gno].v.xv1 > g[gno].v.xv2);
}

int isyreversed(int gno)
{
    return (g[gno].v.yv1 > g[gno].v.yv2);
}

int islogx(int gno)
{
    return (g[gno].xscale == SCALE_LOG);
}

int islogy(int gno)
{
    return (g[gno].yscale == SCALE_LOG);
}

char *graph_types(int it)
{
    static char s[16];

    switch (it) {
    case GRAPH_XY:
	strcpy(s, "XY");
	break;
    case GRAPH_CHART:
	strcpy(s, "Chart");
	break;
    case GRAPH_POLAR:
	strcpy(s, "Polar");
	break;
    case GRAPH_SMITH:
	strcpy(s, "Smith");
	break;
    default:
        strcpy(s, "Unknown");
	break;
   }
    return s;
}

char *scale_types(int it)
{
    static char s[16];

    switch (it) {
    case SCALE_NORMAL:
	strcpy(s, "Normal");
	break;
    case SCALE_LOG:
	strcpy(s, "Logarithmic");
	break;
    case SCALE_REC:
	strcpy(s, "Reciprocal");
	break;
    default:
        strcpy(s, "Unknown");
	break;
    }
    
    return s;
}

int get_format_index(int f)
{
    int i = 0;

    while (f != format_types[i] && format_types[i] != FORMAT_INVALID) {
	i++;
    }
    return i;
}

char *get_format_types(int f)
{
    static char s[128];

    strcpy(s, "decimal");
    switch (f) {
    case FORMAT_DECIMAL:
	strcpy(s, "decimal");
	break;
    case FORMAT_EXPONENTIAL:
	strcpy(s, "exponential");
	break;
    case FORMAT_POWER:
	strcpy(s, "power");
	break;
    case FORMAT_GENERAL:
	strcpy(s, "general");
	break;
    case FORMAT_DDMMYY:
	strcpy(s, "ddmmyy");
	break;
    case FORMAT_MMDDYY:
	strcpy(s, "mmddyy");
	break;
    case FORMAT_MMYY:
	strcpy(s, "mmyy");
	break;
    case FORMAT_MMDD:
	strcpy(s, "mmdd");
	break;
    case FORMAT_MONTHDAY:
	strcpy(s, "monthday");
	break;
    case FORMAT_DAYMONTH:
	strcpy(s, "daymonth");
	break;
    case FORMAT_MONTHS:
	strcpy(s, "months");
	break;
    case FORMAT_MONTHSY:
	strcpy(s, "monthsy");
	break;
    case FORMAT_MONTHL:
	strcpy(s, "monthl");
	break;
    case FORMAT_DAYOFWEEKS:
	strcpy(s, "dayofweeks");
	break;
    case FORMAT_DAYOFWEEKL:
	strcpy(s, "dayofweekl");
	break;
    case FORMAT_DAYOFYEAR:
	strcpy(s, "dayofyear");
	break;
    case FORMAT_HMS:
	strcpy(s, "hms");
	break;
    case FORMAT_MMDDHMS:
	strcpy(s, "mmddhms");
	break;
    case FORMAT_MMDDYYHMS:
	strcpy(s, "mmddyyhms");
	break;
    case FORMAT_DEGREESLON:
	strcpy(s, "degreeslon");
	break;
    case FORMAT_DEGREESMMLON:
	strcpy(s, "degreesmmlon");
	break;
    case FORMAT_DEGREESMMSSLON:
	strcpy(s, "degreesmmsslon");
	break;
    case FORMAT_MMSSLON:
	strcpy(s, "mmsslon");
	break;
    case FORMAT_DEGREESLAT:
	strcpy(s, "degreeslat");
	break;
    case FORMAT_DEGREESMMLAT:
	strcpy(s, "degreesmmlat");
	break;
    case FORMAT_DEGREESMMSSLAT:
	strcpy(s, "degreesmmsslat");
	break;
    case FORMAT_MMSSLAT:
	strcpy(s, "mmsslat");
	break;
    }
    return s;
}

void kill_graph(int gno)
{
    int i, j;

    if (gno == maxgraph) {
	for (i = 0; i < maxgraph; i++) {
	    for (j = 0; j < g[i].maxplot; j++) {
		killset(i, j);
	    }
	    set_default_graph(i);
	}
    } else {
	for (i = 0; i < g[gno].maxplot; i++) {
	    killset(gno, i);
	}
	set_default_graph(gno);
    }
}

void copy_graph(int from, int to)
{
    int i, j;
    plotarr *p;
    kill_graph(to);
    p = g[to].p;
    memcpy(&g[to], &g[from], sizeof(graph));

    g[to].labs.title = copy_plotstr(g[from].labs.title);
    g[to].labs.stitle = copy_plotstr(g[from].labs.stitle);

    for (j = 0; j < MAXAXES; j++) {
	g[to].t[j].label = copy_plotstr(g[from].t[j].label);
	for (i = 0; i < MAX_TICKS; i++) {
	    if (g[from].t[j].t_speclab[i].s != NULL) {
		g[to].t[j].t_speclab[i] = copy_plotstr(g[from].t[j].t_speclab[i]);
	    }
	}
    }

    g[to].p = p;
    set_graph_active(to);	/* TODO compare maxplots */
    for (i = 0; i < g[from].maxplot; i++) {
	for (j = 0; j < MAX_SET_COLS; j++) {
	    g[to].p[i].ex[j] = NULL;
	}
	g[to].p[i].active = FALSE;
	if (isactive_set(from, i)) {
	    do_copyset(from, i, to, i);
	}
    }
}

void copy_graph_sets_only(int from, int to)
{
    int i, j;
    kill_graph(to);
    set_graph_active(to);	/* TODO compare maxplots */
    for (i = 0; i < g[from].maxplot; i++) {
	for (j = 0; j < MAX_SET_COLS; j++) {
	    g[to].p[i].ex[j] = NULL;
	}
	g[to].p[i].active = FALSE;
	if (isactive_set(from, i)) {
	    do_copyset(from, i, to, i);
	}
    }
}

void swap_graph(int from, int to)
{
    graph gtmp;
    memcpy(&gtmp, &g[from], sizeof(graph));
    memcpy(&g[from], &g[to], sizeof(graph));
    memcpy(&g[to], &gtmp, sizeof(graph));
    set_dirtystate();
}

void get_graph_box(int i, boxtype * b)
{
    memcpy(b, &boxes[i], sizeof(boxtype));
}

void get_graph_ellipse(int i, ellipsetype * b)
{
    memcpy(b, &ellip[i], sizeof(ellipsetype));
}

void get_graph_line(int i, linetype * l)
{
    memcpy(l, &lines[i], sizeof(linetype));
}

void get_graph_string(int i, plotstr * s)
{
    memcpy(s, &pstr[i], sizeof(plotstr));
}

void get_graph_framep(int gno, framep * f)
{
    memcpy(f, &g[gno].f, sizeof(framep));
}

void get_graph_world(int gno, world * w)
{
    memcpy(w, &g[gno].w, sizeof(world));
}

void get_graph_view(int gno, view * v)
{
    memcpy(v, &g[gno].v, sizeof(view));
}

void get_graph_labels(int gno, labels * labs)
{
    memcpy(labs, &g[gno].labs, sizeof(labels));
}

void get_graph_plotarr(int gno, int i, plotarr * p)
{
    memcpy(p, &g[gno].p[i], sizeof(plotarr));
}

void get_graph_tickmarks(int gno, tickmarks * t, int a)
{
    memcpy(t, &g[gno].t[a], sizeof(tickmarks));
}

void get_graph_legend(int gno, legend * leg)
{
    memcpy(leg, &g[gno].l, sizeof(legend));
}

/*
 *
 * set graph props
 *
 */

void set_graph_box(int i, boxtype * b)
{
    memcpy(&boxes[i], b, sizeof(boxtype));
}

void set_graph_line(int i, linetype * l)
{
    memcpy(&lines[i], l, sizeof(linetype));
}

void set_graph_string(int i, plotstr * s)
{
    memcpy(&pstr[i], s, sizeof(plotstr));
}

void set_graph_active(int gno)
{

    g[gno].active = TRUE;
}

void set_graph_framep(int gno, framep * f)
{
    memcpy(&g[gno].f, f, sizeof(framep));
}

void set_graph_world(int gno, world * w)
{
    memcpy(&g[gno].w, w, sizeof(world));
}

void set_graph_view(int gno, view * v)
{
    memcpy(&g[gno].v, v, sizeof(view));
}

void set_graph_viewport(int gno, double vx1, double vy1, double vx2, double vy2)
{
    if (isxreversed(gno)) {
	g[gno].v.xv2 = vx1;
	g[gno].v.xv1 = vx2;
    } else {
	g[gno].v.xv1 = vx1;
	g[gno].v.xv2 = vx2;
    }
    if (isyreversed(gno)) {
	g[gno].v.yv2 = vy1;
	g[gno].v.yv1 = vy2;
    } else {
	g[gno].v.yv1 = vy1;
	g[gno].v.yv2 = vy2;
    }
    set_dirtystate();
}

void set_graph_labels(int gno, labels * labs)
{
    memcpy(&g[gno].labs, labs, sizeof(labels));
}

void set_graph_plotarr(int gno, int i, plotarr * p)
{
    memcpy(&g[gno].p[i], p, sizeof(plotarr));
}

void set_graph_tickmarks(int gno, tickmarks * t, int a)
{
    memcpy(&g[gno].t[a], t, sizeof(tickmarks));
}

void set_graph_legend(int gno, legend * leg)
{
    memcpy(&g[gno].l, leg, sizeof(legend));
}



int wipeout(void)
{
    if (!noask && is_dirtystate()) {
        if (!yesno("Abandon unsaved project?", NULL, NULL, NULL)) {
            return 1;
        }
    }
    kill_graph(maxgraph);
    do_clear_lines();
    do_clear_boxes();
    do_clear_ellipses();
    do_clear_text();
    cg = 0;
    *description = 0;
    strcpy(docname, NONAME);;
    clear_dirtystate();
    return 0;
}


/* --- The following routines determine default scaling and tickmarks --- */

void autoscale_axis_byset(int gno, int axistype, int setno)
{
    double xmax, xmin, ymax, ymin, extra_range;
    double x1, x2, y1, y2;
    int i, first = TRUE;

    if (g[gno].type == GRAPH_SMITH) {
        if (axistype == AXIS_TYPE_X || axistype == AXIS_TYPE_ANY) {
            g[gno].w.xg1 = -1.0;
            g[gno].w.yg1 = -1.0;
        }
        if (axistype == AXIS_TYPE_Y || axistype == AXIS_TYPE_ANY) {
            g[gno].w.xg2 = 1.0;
            g[gno].w.yg2 = 1.0;
	}
        return;
    }

    if (setno == ALL_SETS) {
        xmax = xmin = ymax = ymin = 0.0;
        for (i = 0; i < g[gno].maxplot; i++) {
            if (isactive_set(gno, i)) {
                getsetminmax(gno, i, &x1, &x2, &y1, &y2);
                if (first) {
                    xmin = x1;
                    xmax = x2;
                    ymin = y1;
                    ymax = y2;
                    first = FALSE;
                } else {
                    xmin = (x1 < xmin) ? x1 : xmin;
                    xmax = (x2 > xmax) ? x2 : xmax;
                    ymin = (y1 < ymin) ? y1 : ymin;
                    ymax = (y2 > ymax) ? y2 : ymax;
                }
            }
        }
    } else {
        getsetminmax(gno, setno, &xmin, &xmax, &ymin, &ymax);
    }

    if (axistype == AXIS_TYPE_X || axistype == AXIS_TYPE_ANY) {
        if (xmin != xmax) {
            extra_range = 0.0;
        } else {
            extra_range = 0.1 * fabs(xmin);
        }
        g[gno].w.xg1 = xmin - extra_range;
        g[gno].w.xg2 = xmax + extra_range;
    }
    
    if (axistype == AXIS_TYPE_Y || axistype == AXIS_TYPE_ANY) {
        if (ymin != ymax) {
            extra_range = 0.0;
        } else {
            extra_range = 0.1 * fabs(ymin);
        }
        g[gno].w.yg1 = ymin - extra_range;
        g[gno].w.yg2 = ymax + extra_range;
    }
    
    if (g[gno].type == GRAPH_CHART) {
	if (axistype == AXIS_TYPE_X || axistype == AXIS_TYPE_ANY) {
            g[gno].w.xg1 -= (g[gno].w.xg2 - g[gno].w.xg1) * 0.05;
	    g[gno].w.xg2 += (g[gno].w.xg2 - g[gno].w.xg1) * 0.05;
	}
        if (axistype == AXIS_TYPE_Y || axistype == AXIS_TYPE_ANY) {
            if (g[gno].w.yg1 > 0.0) {
	        g[gno].w.yg1 = 0.0;
	    }
        }
    }
}

int axis_type(int axis)
{
    int atype;
    
    switch (axis) {
    case ALL_AXES:
        atype = AXIS_TYPE_ANY;
        break;
    case ALL_X_AXES:
    case X_AXIS:
    case ZX_AXIS:
        atype = AXIS_TYPE_X;
        break;
    case ALL_Y_AXES:
    case Y_AXIS:
    case ZY_AXIS:
        atype = AXIS_TYPE_Y;
        break;
    default:
        atype = AXIS_TYPE_BAD;
        break;
    }
    
    return atype;
}

void autotick_axis(int gno, int axis)
{
    switch (axis) {
    case ALL_AXES:
        default_axis(gno, g[gno].auto_type, X_AXIS);
        default_axis(gno, g[gno].auto_type, ZX_AXIS);
        default_axis(gno, g[gno].auto_type, Y_AXIS);
        default_axis(gno, g[gno].auto_type, ZY_AXIS);
        break;
    case ALL_X_AXES:
        default_axis(gno, g[gno].auto_type, X_AXIS);
        default_axis(gno, g[gno].auto_type, ZX_AXIS);
        break;
    case ALL_Y_AXES:
        default_axis(gno, g[gno].auto_type, Y_AXIS);
        default_axis(gno, g[gno].auto_type, ZY_AXIS);
        break;
    default:
        default_axis(gno, g[gno].auto_type, axis);
        break;
    }

}

void autoscale_set(int gno, int setno, int axis)
{
    if (setno == ALL_SETS || isactive_set(gno, setno)) {
	autoscale_axis_byset(gno, axis_type(axis), setno);
	autotick_axis(gno, axis);
#ifndef NONE_GUI
        update_all(gno);
#endif
    }
}


static void default_axis(int gno, int method, int axis)
{
    world w;
    double llim, ulim;
    
/*
 *     get_graph_tickmarks(gno, &t, axis);
 *     
 *     t.tmajor = (ulim - llim) / t.t_autonum;
 *     cx = (int) (log10(t.tmajor));
 * 
 *     newprec = ((cx < 0) ? -cx + 1 : t.tl_prec);
 *     if (t.tl_format != FORMAT_GENERAL || newprec > t.tl_prec) {
 *         t.tl_prec = newprec;
 *         if (t.tl_prec > 9) {
 *             t.tl_prec = 2;
 *             t.tl_format = FORMAT_EXPONENTIAL;
 *         }
 *     }
 * 
 *     set_graph_tickmarks(gno, &t, axis);
 */

    if (method == TYPE_AUTO) {
        get_graph_world(gno, &w);
        if (is_xaxis(axis)) {
            llim = w.xg1;
            ulim = w.xg2;
        } else {
            llim = w.yg1;
            ulim = w.yg2;
        }
	auto_ticks(gno, axis, &llim, &ulim);
	if (is_xaxis(axis)) {
	    w.xg1 = llim;
	    w.xg2 = ulim;
	} else {
	    w.yg1 = llim;
	    w.yg2 = ulim;
	}
	set_graph_world(gno, &w);
    }
}


#define USE_HECKBERT

#ifndef USE_HECKBERT

/*
 * Graph axis auto-scaling (Nicer minor-ticks than "Heckbert")
 *
 * Dan Hofferth, 10 Mar 97
 */

static void auto_ticks(int gno, int axis, double *gmin, double *gmax)
{
    tickmarks t;
    view v;
    double imax = *gmax, imin = *gmin, idiv, xdiv, dv;
    double iinc, xxp, norm, oinc, sdiv, omin, omax, odiv, sinc;

    get_graph_tickmarks(gno, &t, axis);
    get_graph_view(gno, &v);

    if (!is_xaxis(axis) && (g[gno].yscale == SCALE_LOG)) {
	imax = ceil(log10(imax));
	imin = floor(log10(imin));
    } else if ((is_xaxis(axis)) && (g[gno].xscale == SCALE_LOG)) {
	imax = ceil(log10(imax));
	imin = floor(log10(imin));
    }
    
    if (is_xaxis(axis)) {
	dv = fabs( v.xv2 - v.xv1 ) / 0.7;
    } else {
	dv = fabs( v.yv2 - v.yv1 ) / 0.7;
    }
    
    idiv =  0.0;
    xdiv = ceil( 10.0 * dv );
    if ( imin == imax ) {
   	imin = imin - 0.5;
   	imax = imax + 0.5;
    }
    if ( imin < 0.0 && imax > 0.0 && xdiv == 1.0 ) {
        xdiv = 2.0;
    }
    while ( -1 ) {
	iinc = ( imax - imin ) / xdiv;
	xxp = pow( 10.0, floor( log10( iinc ) ) );
   	norm = iinc / xxp;
   	if ( norm > 5.0 ) {
	    oinc = 10.0 * xxp;
	    sdiv = 5.0;
	}
	else if ( norm > 4.0 ) {
	    oinc = 5.0  * xxp;
	    sdiv = 5.0;
	} else if ( norm > 2.0 ) {
            oinc = 4.0  * xxp;
            sdiv = 4.0;
        } else if ( norm > 1.0 ) {
            oinc = 2.0  * xxp;
            sdiv = 2.0;
        } else {
            oinc = 1.0  * xxp;
            sdiv = 5.0;
        }
	omin = floor( imin / oinc ) * oinc;
   	omax = omin + xdiv * oinc;
   	if ( omin > imin || omax < imax ) {
      	    imin = (( imin < omin )? imin : omin );
      	    imax = (( imax > omax )? imax : omax );
      	    continue;
   	}
   	if ( idiv == 0.0 ) {
      	    odiv = ( omax - imax ) / oinc;
      	    odiv = ( odiv >= 0 ) ? floor( odiv ) : ceil( odiv );
      	    if ( odiv > 0 ) {
		/* imin = omin */
		xdiv = xdiv - odiv;
		continue;
	    }
	}
	break;
    }
    sinc = oinc / sdiv;
    
    if (!is_xaxis(axis) && (g[gno].type == GRAPH_LOGY || g[gno].type == GRAPH_LOGXY)) {
	*gmax = pow(10.0, omax);
	*gmin = pow(10.0, omin);
	t.tmajor = (int) oinc;
	if (t.tmajor == 0.0) {
	    t.tmajor = 1.0;
	}
	if ((int) t.tmajor < 2) {
	    t.tminor = 1.0;
	} else {
	    t.tminor = 0.0;
	}
	if (fabs(omax) > 6.0 || fabs(omin) > 6.0) {
	    t.tl_format = FORMAT_POWER;
	    t.tl_prec = 0;
	} else {
	    t.tl_format = FORMAT_DECIMAL;
	    t.tl_prec = 0;
	}
    } else if ((is_xaxis(axis)) && 
               (g[gno].type == GRAPH_LOGX || g[gno].type == GRAPH_LOGXY)) {
	*gmax = pow(10.0, omax);
	*gmin = pow(10.0, omin);
	t.tmajor = (int) oinc;
	if (t.tmajor == 0.0) {
	    t.tmajor = 1.0;
	}
	if (fabs(omax) > 6.0 || fabs(omin) > 6.0) {
	    t.tl_format = FORMAT_POWER;
	    t.tl_prec = 0;
	} else {
	    t.tl_format = FORMAT_DECIMAL;
	    t.tl_prec = 0;
	}
	if ((int) t.tmajor < 2) {
	    t.tminor = 1.0;
	} else {
	    t.tminor = 0.0;
	}
    } else {
	*gmax = omax;
	*gmin = omin;
	t.tmajor = oinc;
	t.tminor = sinc;
    }
    set_graph_tickmarks(gno, &t, axis);
}

#else /* do USE_HECKBERT */

/*
 * label: test program to demonstrate nice graph axis labeling
 *
 * Paul Heckbert, 2 Dec 88
 */

static double nicenum(double x, int round);

static void auto_ticks(int gno, int axis, double *gmin, double *gmax)
{
    tickmarks t;
    double range, d, tmpmax = *gmax, tmpmin = *gmin;

    get_graph_tickmarks(gno, &t, axis);
    if (!is_xaxis(axis) && (g[gno].yscale == SCALE_LOG)) {
	tmpmax = ceil(log10(tmpmax));
	tmpmin = floor(log10(tmpmin));
    } else if ((is_xaxis(axis)) && (g[gno].xscale == SCALE_LOG)) {
	tmpmax = ceil(log10(tmpmax));
	tmpmin = floor(log10(tmpmin));
    }
    range = nicenum(tmpmax - tmpmin, 0);
    d = nicenum(range / (t.t_autonum - 1), 1);
    tmpmin = floor(tmpmin / d) * d;
    tmpmax = ceil(tmpmax / d) * d;
    if ((is_xaxis(axis) && (g[gno].xscale == SCALE_LOG)) ||
        (!is_xaxis(axis) && (g[gno].yscale == SCALE_LOG))) {
	*gmax = pow(10.0, tmpmax);
	*gmin = pow(10.0, tmpmin);
	t.tmajor = (int) d;
	if (t.tmajor == 0.0) {
	    t.tmajor = 1.0;
	}
/*
 *         t.tmajor = pow(10.0, t.tmajor);
 * 	if (fabs(tmpmax) > 6.0 || fabs(tmpmin) > 6.0) {
 * 	    t.tl_format = FORMAT_POWER;
 * 	    t.tl_prec = 0;
 * 	} else {
 * 	    t.tl_format = FORMAT_DECIMAL;
 * 	    t.tl_prec = 0;
 * 	}
 */
    } else {
	*gmax = tmpmax;
	*gmin = tmpmin;
	t.tmajor = d;
	t.nminor = 1;
    }
    set_graph_tickmarks(gno, &t, axis);
}

/*
 * nicenum: find a "nice" number approximately equal to x
 * round if round=1, ceil if round=0
 */

static double nicenum(double x, int round)
{
    int exp;
    double f, y;

    exp = floor(log10(x));
    f = x / pow(10., (double) exp);	/* fraction between 1 and 10 */
    if (round)
	if (f < 1.5)
	    y = 1.;
	else if (f < 3.)
	    y = 2.;
	else if (f < 7.)
	    y = 5.;
	else
	    y = 10.;
    else if (f <= 1.)
	y = 1.;
    else if (f <= 2.)
	y = 2.;
    else if (f <= 5.)
	y = 5.;
    else
	y = 10.;
    return y * pow(10., (double) exp);
}

#endif
