/*
 * GRACE PDF driver
 */

#include <config.h>
#include <cmath.h>

#include <stdio.h>
#include <unistd.h>
#include <string.h>

#include "defines.h"
#include "globals.h"
#include "utils.h"
#include "draw.h"
#include "device.h"
#include "patterns.h"
#include "t1fonts.h"
#include "pdfdrv.h"

#include "patchlevel.h"

#include "protos.h"

/* uncomment this if the PDFlib was compiled w/ support for TIFF images */
/* #define USE_TIFF */
#include <pdf.h>

static unsigned long page_scale;
static float page_scalef;

extern FILE *prstream;

static PDF *phandle;

int pdfinitgraphics(void)
{
    Page_geometry pg;
    PDF_info *info;
    static char buf[GR_MAXPATHLEN];
    
    /* device-dependent routines */
    devsetpen       = pdf_setpen;
    devsetline      = pdf_setlinestyle;
    devsetlinew     = pdf_setlinewidth;
    
    devupdatecmap   = NULL;
    
    devdrawpolyline = pdf_drawpolyline;
    devdrawpolygon  = pdf_drawpolygon;
    devdrawarc      = pdf_drawarc;
    devfillarc      = pdf_fillarc;
    devputpixmap    = pdf_putpixmap;
    devputtext      = pdf_puttext;
    
    devleavegraphics = pdf_leavegraphics;
    
    pg = get_page_geometry();
    
    page_scale = MIN2(pg.height, pg.width);
    page_scalef = (float) page_scale*72.0/pg.dpi_x;

    info = PDF_get_info();
    
    sprintf(buf, "Grace v%d.%d.%d %s\n",
                                MAJOR_REV, MINOR_REV, PATCHLEVEL, BETA_VER);
    info->Creator = buf;
    info->Author = getlogin();
    info->Title = docname;
    
    sprintf(buf, "%s/fonts/type1", grace_home);
    info->fontpath = buf;
        
    phandle = PDF_open(prstream, info);
    
    if (phandle == NULL) {
        return GRACE_EXIT_FAILURE;
    }
    
    PDF_begin_page(phandle, pg.width*72.0/pg.dpi_x, pg.height*72.0/pg.dpi_y);
    PDF_scale(phandle, page_scalef, page_scalef);
    
    return GRACE_EXIT_SUCCESS;
}

void pdf_setpen(void)
{
    Pen pen;
    RGB *rgb;
    
    pen = getpen();
    rgb = get_rgb(pen.color);
    PDF_setrgbcolor(phandle, (float) rgb->red/(MAXCOLORS - 1),
                             (float) rgb->green/(MAXCOLORS - 1),
                             (float) rgb->blue/(MAXCOLORS - 1)  );     
    /* TODO: patterns */
}

void pdf_setlinestyle(int lines)
{
    int i;
    float lw = (float) getlinewidth();
    float *darray = NULL;
    
    if (lines == 0) {
        PDF_setpolydash(phandle, darray, 0);
    } else if (lines == 1) {
        PDF_setpolydash(phandle, darray, 0); /* length == 0,1 means solid line */
    } else {
        darray = (float *) malloc(dash_array_length[lines]*SIZEOF_FLOAT);
        for (i = 0; i < dash_array_length[lines]; i++) {
            darray[i] = lw*dash_array[lines][i];
        }
        PDF_setpolydash(phandle, darray, dash_array_length[lines]);
        free (darray);
    }
}

void pdf_setlinewidth(double linew)
{
    PDF_setlinewidth(phandle, (float) linew);
}

void pdf_drawpolyline(VPoint *vps, int n, int mode)
{
    int i;
    
    if (getlinestyle() == 0) {
        return;
    }
    
    PDF_moveto(phandle, (float) vps[0].x, (float) vps[0].y);
    for (i = 1; i < n; i++) {
        PDF_lineto(phandle, (float) vps[i].x, (float) vps[i].y);
    }
    if (mode == POLYLINE_CLOSED) {
        PDF_closepath_stroke(phandle);
    } else {
        PDF_stroke(phandle);
    }
}

void pdf_drawpolygon(VPoint *vps, int nc)
{
    int i;
    
    if (getpattern() == 0) {
        return;
    }
    
    PDF_moveto(phandle, (float) vps[0].x, (float) vps[0].y);
    for (i = 1; i < nc; i++) {
        PDF_lineto(phandle, (float) vps[i].x, (float) vps[i].y);
    }
    PDF_fill(phandle);
}

void pdf_drawarc(VPoint vp1, VPoint vp2, int a1, int a2)
{
    VPoint vpc;
    double rx, ry;
    
    if (getlinestyle() == 0) {
        return;
    }
    
    vpc.x = (vp1.x + vp2.x)/2;
    vpc.y = (vp1.y + vp2.y)/2;
    rx = fabs(vp2.x - vp1.x)/2;
    ry = fabs(vp2.y - vp1.y)/2;
    
    if (rx == 0.0 || ry == 0.0) {
        return;
    }
    
    PDF_save(phandle);
    PDF_scale(phandle, 1.0, ry/rx);
    PDF_moveto(phandle, (float) vpc.x + rx*cos(M_PI*a1), 
                        (float) rx/ry*vpc.y + rx*sin(M_PI*a1));
    PDF_arc(phandle, (float) vpc.x, (float) rx/ry*vpc.y, rx, 
                                        (float) a1, (float) (a1 +a2));
    PDF_stroke(phandle);
    PDF_restore(phandle);
}

void pdf_fillarc(VPoint vp1, VPoint vp2, int a1, int a2)
{
    VPoint vpc;
    double rx, ry;
    
    if (getpattern() == 0) {
        return;
    }
    
    vpc.x = (vp1.x + vp2.x)/2;
    vpc.y = (vp1.y + vp2.y)/2;
    rx = fabs(vp2.x - vp1.x)/2;
    ry = fabs(vp2.y - vp1.y)/2;
    
    if (rx == 0.0 || ry == 0.0) {
        return;
    }
    
    PDF_save(phandle);
    PDF_scale(phandle, 1.0, ry/rx);
    PDF_moveto(phandle, (float) vpc.x + rx*cos(M_PI*a1), 
                        (float) rx/ry*vpc.y + rx*sin(M_PI*a1));
    PDF_arc(phandle, (float) vpc.x, (float) rx/ry*vpc.y, rx, 
                                        (float) a1, (float) (a1 +a2));
    PDF_fill(phandle);
    PDF_restore(phandle);
}

/* TODO: transparent pixmaps */
void pdf_putpixmap(VPoint vp, int width, int height, char *databits, 
                             int pixmap_bpp, int bitmap_pad, int pixmap_type)
{
    byte *buf, *bp;
    long buflen;
    PDF_image image;
    int cindex;
    RGB *rgb, *fg, *bg;
    int	i, k, j;
    long paddedW;

    image.width		= width;
    image.height	= height;
    image.bpc		= 8;
    image.components    = 3;
    image.colorspace    = DeviceRGB;
    
/*
 *     if (pixmap_bpp == 1) {
 *         image.components	= 1;
 *         image.colorspace	= DeviceGray;
 *     } else {
 *         image.components	= 3;
 *         image.colorspace	= DeviceRGB;
 *     }
 */

    buflen = image.width * image.height * image.components;
    
    buf = (byte *) malloc(buflen);
    if (buf == NULL) {
        errmsg("malloc failed in pdf_putpixmap()");
        return;
    }
    
    bp = buf;
    if (pixmap_bpp == 1) {
        paddedW = PAD(width, bitmap_pad);
        fg = get_rgb(getcolor());
        bg = get_rgb(getbgcolor());
        for (k = 0; k < image.height; k++) {
            for (j = 0; j < paddedW/bitmap_pad; j++) {
                for (i = 0; i < bitmap_pad && j*bitmap_pad + i < image.width; i++) {
                    if (bin_dump(&(databits)[k*paddedW/bitmap_pad+j], i, bitmap_pad)) {
                        *bp++ = (byte) (256 * fg->red/(MAXCOLORS - 1));
                        *bp++ = (byte) (256 * fg->green/(MAXCOLORS - 1));
                        *bp++ = (byte) (256 * fg->blue/(MAXCOLORS - 1));
                    } else {
                        *bp++ = (byte) (256 * bg->red/(MAXCOLORS - 1));
                        *bp++ = (byte) (256 * bg->green/(MAXCOLORS - 1));
                        *bp++ = (byte) (256 * bg->blue/(MAXCOLORS - 1));
                    }
                }
            }
        }
    } else {
        for (k = 0; k < image.height; k++) {
            for (j = 0; j < image.width; j++) {
                cindex = (databits)[k*image.width+j];
                rgb = get_rgb(cindex);
                *bp++ = (byte) (256 * rgb->red/(MAXCOLORS - 1));
                *bp++ = (byte) (256 * rgb->green/(MAXCOLORS - 1));
                *bp++ = (byte) (256 * rgb->blue/(MAXCOLORS - 1));
            }
        }
    }
    
    PDF_save(phandle);

    PDF_translate(phandle, vp.x, vp.y);
    PDF_scale(phandle, 1.0/page_scale, 1.0/page_scale);
    PDF_translate(phandle, 0.0, - (float) image.height);
     
    PDF_data_source_from_buf(phandle, &image.src, buf, buflen);
    PDF_place_inline_image(phandle, &image, 0.0, 0.0, 1.0);
    
    PDF_restore(phandle);

    free(buf);
}

void pdf_puttext(VPoint start, VPoint end, double size, 
                                            CompositeString *cstring)
{
    int iglyph;
    float angle;
    float length, pdfstring_length;
    char *fontname, *encscheme;
    int pdflibenc;
    
    size /= page_scalef;
    
    angle = (float) (180.0/M_PI) * atan2(end.y - start.y, end.x - start.x);
    length = (float) hypot (end.x - start.x, end.y - start.y);
    
    pdfstring_length = 0.0;
    
    iglyph = 0;
    while (cstring[iglyph].s != NULL) {
        fontname = get_fontalias(cstring[iglyph].font);
        encscheme = get_encodingscheme(cstring[iglyph].font);
        if (strcmp(encscheme, "ISOLatin1Encoding") == 0) {
            pdflibenc = winansi;
        } else if (strcmp(encscheme, "FontSpecific") == 0) {
            pdflibenc = builtin;
        } else {
            pdflibenc = pdfdoc;
        }
        PDF_set_font(phandle, fontname, (float) size*cstring[iglyph].scale, pdflibenc);
        pdfstring_length += PDF_stringwidth(phandle, (unsigned char*) cstring[iglyph].s);
        iglyph++;
    }

    PDF_save(phandle);
    PDF_translate(phandle, (float) start.x, (float) start.y);
    PDF_rotate(phandle, angle);
    /*
     * Compensate for diffs between PDFlib & T1lib 
     * (should Y be scaled the same??)
     */
    PDF_scale(phandle, (float) size*length/pdfstring_length, (float) size);

    PDF_set_text_pos(phandle, 0.0, 0.0);
    
    iglyph = 0;
    while (cstring[iglyph].s != NULL) {
        fontname = get_fontalias(cstring[iglyph].font);
        encscheme = get_encodingscheme(cstring[iglyph].font);
        if (strcmp(encscheme, "ISOLatin1Encoding") == 0) {
            pdflibenc = winansi;
        } else if (strcmp(encscheme, "FontSpecific") == 0) {
            pdflibenc = builtin;
        } else {
            pdflibenc = pdfdoc;
        }
        PDF_set_font(phandle, fontname, (float) cstring[iglyph].scale, pdflibenc);
        if (cstring[iglyph].vshift != 0.0) {
            PDF_translate(phandle, 0.0, (float) cstring[iglyph].vshift);
            PDF_show(phandle, cstring[iglyph].s);
            PDF_translate(phandle, 0.0, - (float) cstring[iglyph].vshift);
        } else {
            PDF_show(phandle, cstring[iglyph].s);
        }
        
        if (cstring[iglyph].underline == TRUE) {
            /* TODO */
        }
        
        if (cstring[iglyph].overline == TRUE) {
            /* TODO */
        }
        
        iglyph++;
    }

    PDF_restore(phandle);
}

void pdf_leavegraphics(void)
{
    PDF_end_page(phandle);
    PDF_close(phandle);
}
