/*
 * XCC - XML Compiler-Compiler
 * 
 * Copyright (c) 2000-2005 Evgeny Stambulchik
 * 
 * 
 *                           All Rights Reserved
 * 
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 * 
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 * 
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * Library routines for XCC executable itself
 */

#include <string.h>
#include <getopt.h>

#include "xccP.h"
#include "bundle.i"
#include "xfile.h"

/* TODO: get rid of it */
#define KEY_SHIFT   1

XCC *xcc_xcc_new(void)
{
    XCC *xcc;
    
    xcc = xcc_malloc(sizeof(XCC));
    memset(xcc, 0, sizeof(XCC));
    
    xcc->a_types = xcc_stack_new((XCC_stack_data_free) atype_free);
    xcc->e_types = xcc_stack_new((XCC_stack_data_free) etype_free);
    xcc->elements = xcc_stack_new((XCC_stack_data_free) element_free);
    
    xcc->preamble = xcc_string_new();
    xcc->postamble = xcc_string_new();
    
    return xcc;
}

void xcc_xcc_free(XCC *xcc)
{
    if (xcc) {
        xcc_stack_free(xcc->a_types);
        xcc_stack_free(xcc->e_types);
        xcc_stack_free(xcc->elements);
        xcc_string_free(xcc->preamble);
        xcc_string_free(xcc->postamble);
        xcc_free(xcc->ns_uri);
        xcc_free(xcc->prefix);
        xcc_free(xcc);
    }
}

AType *atype_new(void)
{
    AType *atype;
    atype = xcc_malloc(sizeof(AType));
    memset(atype, 0, sizeof(AType));
    return atype;
}

void atype_free(AType *atype)
{
    if (atype) {
        xcc_free(atype->name);
        xcc_free(atype->ctype);
        xcc_free(atype->ccode);
        xcc_free(atype);
    }
}

EType *etype_new(void)
{
    EType *etype;
    etype = xcc_malloc(sizeof(EType));
    memset(etype, 0, sizeof(EType));
    return etype;
}

void etype_free(EType *etype)
{
    if (etype) {
        xcc_free(etype->name);
        xcc_free(etype->ctype);
        xcc_free(etype->ccode);
        xcc_free(etype);
    }
}

Element *element_new(void)
{
    Element *e;
    e = xcc_malloc(sizeof(Element));
    memset(e, 0, sizeof(Element));
    e->attributes = xcc_stack_new((XCC_stack_data_free) attribute_free);
    e->children   = xcc_stack_new((XCC_stack_data_free) child_free);
    e->data       = xcc_string_new();
    e->same_parents = 1;
    return e;
}

void element_free(Element *e)
{
    if (e) {
        xcc_free(e->name);
        xcc_stack_free(e->attributes);
        xcc_stack_free(e->children);
        xcc_string_free(e->data);
        xcc_free(e);
    }
}

Attribute *attribute_new(void)
{
    Attribute *a;
    a = xcc_malloc(sizeof(Attribute));
    memset(a, 0, sizeof(Attribute));
    return a;
}

void attribute_free(Attribute *a)
{
    if (a) {
        xcc_free(a->name);
        xcc_free(a->ccode);
        xcc_free(a);
    }
}

Child *child_new(void)
{
    Child *c;
    c = xcc_malloc(sizeof(Child));
    memset(c, 0, sizeof(Child));
    return c;
}

void child_free(Child *c)
{
    if (c) {
        xcc_free(c->name);
        xcc_free(c->ccode);
        xcc_free(c);
    }
}

AType *get_atype_by_name(XCCStack *a_types, const char *name)
{
    int i, n_atypes = xcc_stack_depth(a_types);
    for (i = 0; i < n_atypes; i++) {
        AType *atype;
        void *p;
        xcc_stack_get_data(a_types, i, &p);
        atype = p;
        if (!strcmp(atype->name, name)) {
            return atype;
        }
    }
    
    return NULL;
}

EType *get_etype_by_name(XCCStack *e_types, const char *name)
{
    int i, n_etypes = xcc_stack_depth(e_types);
    for (i = 0; i < n_etypes; i++) {
        EType *etype;
        void *p;
        xcc_stack_get_data(e_types, i, &p);
        etype = p;
        if (!strcmp(etype->name, name)) {
            return etype;
        }
    }
    
    return NULL;
}


static int output_header(FILE *fp)
{
    fprintf(fp, "/* Generated by %s */\n\n", xcc_get_version_string());
    fprintf(fp, "#include <xcc.h>\n");
    return XCC_RETURN_SUCCESS;
}

static int output_bundle(FILE *fp)
{
    int i = 0;
    char *s;
    while ((s = bundle_str[i])) {
        fprintf(fp, "%s\n", s);
        i++;
    }
    
    return XCC_RETURN_SUCCESS;
}


static int output_preamble(const XCC *xcc, FILE *fp)
{
    if (xcc->preamble->s) {
        fprintf(fp, "%s\n", xcc->preamble->s);
    }
    return XCC_RETURN_SUCCESS;
}

static int output_postamble(const XCC *xcc, FILE *fp)
{
    if (xcc->postamble->s) {
        fprintf(fp, "%s\n", xcc->postamble->s);
    }
    return XCC_RETURN_SUCCESS;
}

static int output_atype_union(const XCC *xcc, FILE *fp)
{
    int i, n_atypes;
    
    n_atypes = xcc_stack_depth(xcc->a_types);
    if (n_atypes > 0) {
        fprintf(fp, "typedef union {\n");
        for (i = 0; i < n_atypes; i++) {
            AType *atype;
            void *p;
            xcc_stack_get_data(xcc->a_types, i, &p);
            atype = p;
            fprintf(fp, "    %s %s;\n", atype->ctype, atype->name);
        }
        fprintf(fp, "} XCCAType;\n\n");
    }
    
    return XCC_RETURN_SUCCESS;
}

static int output_etype_union(const XCC *xcc, FILE *fp)
{
    int i, n_etypes;
    
    n_etypes = xcc_stack_depth(xcc->e_types);
    fprintf(fp, "typedef union {\n");
    for (i = 0; i < n_etypes; i++) {
        EType *etype;
        void *p;
        xcc_stack_get_data(xcc->e_types, i, &p);
        etype = p;
        fprintf(fp, "    %s %s;\n", etype->ctype, etype->name);
    }
    fprintf(fp, "    void * unicast;\n");
    fprintf(fp, "} XCCEType;\n\n");
    
    return XCC_RETURN_SUCCESS;
}

static char *replace(const char *str, const char *f, const char *r)
{
    int inlen, outlen, flen, rlen, diff;
    char *p, *rp, *p_old;
    char *retval;
    
    inlen  = strlen(str);
    flen   = strlen(f);
    rlen   = strlen(r);
    
    outlen = inlen + 1;
    
    p = strstr(str, f);
    while (p) {
        outlen += (rlen - flen);
        p += flen;
        p = strstr(p, f);
    }
    
    retval = xcc_malloc(outlen);
    rp = retval;
    
    p_old = (char *) str;
    p = strstr(str, f);
    while (p) {
        diff = p - p_old;
        if (diff) {
            memcpy(rp, p_old, diff);
            rp += diff;
        }
        memcpy(rp, r, rlen);
        rp += rlen;
        
        p  += flen;
        p_old = p;
        
        p = strstr(p_old, f);
    }
    
    diff = inlen - (p_old - str);
    if (diff) {
        memcpy(rp, p_old, diff);
    }
    
    retval[outlen - 1] = '\0';
    
    return retval;
}

static char *print_sharp_name(const char *name)
{
    char *pname;
    
    if (!name) {
        pname = xcc_strdup("NULL");
    } else
    if (name[0] == '#') {
        pname = xcc_strdup(name + 1);
    } else {
        pname = xcc_malloc(strlen(name) + 3);
        if (pname) {
            sprintf(pname, "\"%s\"", name);
        }
    }
    
    return pname;
}

static int output_element_tab(const XCC *xcc, FILE *fp)
{
    int i, n_elements;
    
    n_elements = xcc_stack_depth(xcc->elements);
    fprintf(fp, "static XCCElementEntry XCCElementTab[] = {\n");
    for (i = 0; i < n_elements; i++) {
        Element *e;
        char *pname;
        void *p;
        xcc_stack_get_data(xcc->elements, i, &p);
        e = p;
        e->id = i + KEY_SHIFT;
        pname = print_sharp_name(e->name);
        fprintf(fp, "    {%d, %s}%s\n", e->id, pname, i == n_elements - 1 ? "":",");
        xcc_free(pname);
    }
    fprintf(fp, "};\n\n");

    fprintf(fp, "static int get_element_id_by_name(const char *name)\n");
    fprintf(fp, "{\n");
    fprintf(fp, "    int i;\n");
    fprintf(fp, "    for (i = 0; i < %d; i++) {\n", n_elements);
    fprintf(fp, "        if (!strcmp(XCCElementTab[i].name, name)) {\n");
    fprintf(fp, "            return XCCElementTab[i].key;\n");
    fprintf(fp, "        }\n");
    fprintf(fp, "    }\n");
    fprintf(fp, "    return -1;\n");
    fprintf(fp, "}\n\n");

    fprintf(fp, "static char *get_element_name_by_id(int id)\n");
    fprintf(fp, "{\n");
    fprintf(fp, "    int i;\n");
    fprintf(fp, "    for (i = 0; i < %d; i++) {\n", n_elements);
    fprintf(fp, "        if (XCCElementTab[i].key == id) {\n");
    fprintf(fp, "            return XCCElementTab[i].name;\n");
    fprintf(fp, "        }\n");
    fprintf(fp, "    }\n");
    fprintf(fp, "    return NULL;\n");
    fprintf(fp, "}\n\n");

    
    return XCC_RETURN_SUCCESS;
}

static Element *get_element_by_name(const XCCStack *elements, const char *name)
{
    int i, n_elements = xcc_stack_depth(elements);
    for (i = 0; i < n_elements; i++) {
        Element *e;
        void *p;
        xcc_stack_get_data(elements, i, &p);
        e = p;
        if (!strcmp(e->name, name)) {
            return e;
        }
    }
    
    return NULL;
}

static int output_init_occurrence(const XCC *xcc, FILE *fp)
{
    int i, n_elements, n_children;
    void *p;
    Element *e;

    n_elements = xcc_stack_depth(xcc->elements);

    fprintf(fp, "static XCCOccurrence *init_occurrence(int element_id)\n");
    fprintf(fp, "{\n");

    fprintf(fp, "    XCCOccurrence *occurrence;\n");

    fprintf(fp, "    switch (element_id) {\n");
    for (i = 0; i < n_elements; i++) {
        xcc_stack_get_data(xcc->elements, i, &p);
        e = p;

        n_children = xcc_stack_depth(e->children);
        if (!n_children) {
            fprintf(fp, "    case %d:\n", e->id);
        }
    }
    fprintf(fp, "        return NULL;\n");
    fprintf(fp, "        break;\n");
    fprintf(fp, "    }\n\n");

    fprintf(fp, "    occurrence = xcc_malloc(%d*sizeof(XCCOccurrence));\n",
        n_elements);
    fprintf(fp, "    if (occurrence) {\n");
    fprintf(fp, "        memset(occurrence, 0, %d*sizeof(XCCOccurrence));\n",
        n_elements);
    fprintf(fp, "        switch (element_id) {\n");

    for (i = 0; i < n_elements; i++) {
        int j;
        
        xcc_stack_get_data(xcc->elements, i, &p);
        e = p;
        n_children = xcc_stack_depth(e->children);

        if (n_children) {
            fprintf(fp, "        case %d:\n", e->id);

            for (j = 0; j < n_children; j++) {
                Child *c;
                Element *ce;
                xcc_stack_get_data(e->children, j, &p);
                c = p;

                ce = get_element_by_name(xcc->elements, c->name);
                if (!ce) {
                    xcc_error("couldn't find definition for element %s", c->name);
                    return XCC_RETURN_FAILURE;
                }

                fprintf(fp, "            occurrence[%d].allowed = 1;\n",
                    ce->id - KEY_SHIFT);
                if (c->minOccurs) {
                    fprintf(fp, "            occurrence[%d].minOccurs = %d;\n",
                        ce->id - KEY_SHIFT, c->minOccurs);
                }
                if (c->maxOccurs) {
                    fprintf(fp, "            occurrence[%d].maxOccurs = %d;\n",
                        ce->id - KEY_SHIFT, c->maxOccurs);
                }
            }

            fprintf(fp, "            break;\n");
        }
    }
    
    fprintf(fp, "        }\n");
    fprintf(fp, "    }\n");


    fprintf(fp, "    return occurrence;\n");
    fprintf(fp, "}\n\n");
    
    return XCC_RETURN_SUCCESS;
}

static int output_start_handler(const XCC *xcc, FILE *fp)
{
    int i, n_elements, n_attributes, n_attributes_max = 0;
    char *pns_uri, *buf1, *buf2;
    Element *e;

    n_elements = xcc_stack_depth(xcc->elements);

    /* find max number of attributes per element */
    for (i = 0; i < n_elements; i++) {
        void *p;
        
        xcc_stack_get_data(xcc->elements, i, &p);
        e = p;
        n_attributes = xcc_stack_depth(e->attributes);
        if (n_attributes > n_attributes_max) {
            n_attributes_max = n_attributes;
        }
    }
    
    fprintf(fp, "static void %s_start_handler(void *data, const char *el, const char **attr)\n",
                xcc->prefix);  
    fprintf(fp, "{\n");
    fprintf(fp, "    XCCParserData *pdata = (XCCParserData *) data;\n");
    fprintf(fp, "    XCCNode *pnode = NULL, *node;\n");
    fprintf(fp, "    XCCEType element;\n");
    fprintf(fp, "    XCCAType attribute;\n");
    fprintf(fp, "    int i, element_id = -1, parent_id = -1, skip = 0, askip;\n");
    fprintf(fp, "    const char *avalue;\n");
    fprintf(fp, "    char *aname, *el_local;\n");
    fprintf(fp, "    char *attribs_required[%d];\n", n_attributes_max);
    fprintf(fp, "    int nattribs_required = 0;\n");
    fprintf(fp, "\n");
    fprintf(fp, "    if (pdata->error) {\n");
    fprintf(fp, "        return;\n");
    fprintf(fp, "    }\n\n");
    
    fprintf(fp, "    memset(attribs_required, 0, %d*sizeof(char *));\n\n",
        n_attributes_max);
    
    fprintf(fp, "    pdata->cbuflen = 0;\n");
    fprintf(fp, "    if (pdata->cbufsize) {\n");
    fprintf(fp, "        pdata->cbuffer[0] = '\\0';\n");
    fprintf(fp, "    }\n\n");

    fprintf(fp, "    element.unicast = NULL;\n\n");

    pns_uri = print_sharp_name(xcc->ns_uri);
    fprintf(fp, "    el_local  = xcc_get_local(el, %s, &skip);\n", pns_uri);
    fprintf(fp, "    if (skip) {\n");
    fprintf(fp, "        goto e_switch;\n");
    fprintf(fp, "    }\n\n");

    fprintf(fp, "    if (xcc_stack_depth(pdata->nodes) == 0) {\n");
    fprintf(fp, "        pnode = NULL;\n");
    fprintf(fp, "        parent_id = 0;\n");
    fprintf(fp, "    } else {\n");
    fprintf(fp, "        void *p;\n");
    fprintf(fp, "        xcc_stack_get_last(pdata->nodes, &p);\n");
    fprintf(fp, "        pnode = p;\n");
    fprintf(fp, "        parent_id = pnode->id;\n");
    fprintf(fp, "    }\n");
    fprintf(fp, "    if (parent_id < 0) {\n");
    fprintf(fp, "        skip = 1;\n");
    fprintf(fp, "        goto e_switch;\n");
    fprintf(fp, "    }\n\n");

    fprintf(fp, "    element_id = get_element_id_by_name(el_local);\n");

    fprintf(fp, "    if (element_id < 0) {\n");
    fprintf(fp, "        if (pdata->exception_handler(XCC_EELEM, el_local, pnode ? pnode->name:NULL, pdata->udata)) {\n");
    fprintf(fp, "            skip = 1;\n");
    fprintf(fp, "        } else {\n");
    fprintf(fp, "            pdata->error = 1;\n");
    fprintf(fp, "        }\n");
    fprintf(fp, "        goto e_switch;\n");
    fprintf(fp, "    }\n\n");
    
    fprintf(fp, "    if (!pnode) {\n");
    fprintf(fp, "        goto e_switch;\n");
    fprintf(fp, "    }\n\n");
    
    fprintf(fp, "    if (pnode->occurrence) {\n");
    fprintf(fp, "        pnode->occurrence[element_id - %d].occurred++;\n",
        KEY_SHIFT);
    fprintf(fp, "    }\n\n");
    
    fprintf(fp, "    if (!pnode->occurrence || !pnode->occurrence[element_id - %d].allowed) {\n",
        KEY_SHIFT);
    fprintf(fp, "        if (pdata->exception_handler(XCC_ECNTX, el_local, pnode ? pnode->name:NULL, pdata->udata)) {\n");
    fprintf(fp, "            skip = 1;\n");
    fprintf(fp, "        } else {\n");
    fprintf(fp, "            pdata->error = 1;\n");
    fprintf(fp, "        }\n");
    fprintf(fp, "        goto e_switch;\n");
    fprintf(fp, "    }\n\n");
    
    fprintf(fp, "    if (pnode->occurrence[element_id - %d].maxOccurs &&\n",
        KEY_SHIFT);
    fprintf(fp, "        pnode->occurrence[element_id - %d].occurred > pnode->occurrence[element_id - %d].maxOccurs) {\n",
        KEY_SHIFT, KEY_SHIFT);
    fprintf(fp, "        if (pdata->exception_handler(XCC_EEMAX, el_local, pnode->name, pdata->udata)) {\n");
    fprintf(fp, "            skip = 1;\n");
    fprintf(fp, "        } else {\n");
    fprintf(fp, "            pdata->error = 1;\n");
    fprintf(fp, "        }\n");
    fprintf(fp, "        goto e_switch;\n");
    fprintf(fp, "    }\n\n");

    fprintf(fp, "e_switch:\n\n");

    fprintf(fp, "    switch (element_id) {\n");
    for (i = 0; i < n_elements; i++) {
        void *p;
        int j, element_id;
        char ebuf[128], abuf[128];
        Attribute *a;
        int nattribs_required = 0;
        
        xcc_stack_get_data(xcc->elements, i, &p);
        e = p;
        element_id  = e->id;

        if (!e->etype) {
            xcc_error("couldn't find element type of %s", e->name);
            return XCC_RETURN_FAILURE;
        }

        sprintf(ebuf, "element.%s", e->etype->name);
        
        fprintf(fp, "    case %d: /* %s */\n", element_id, e->name);
        buf1 = replace(e->etype->ccode, "$$", ebuf);
        buf2 = replace(buf1, "$U", "pdata->udata");
        xcc_free(buf1);
        if (e->parent_etype) {
            char buf[128];
            sprintf(buf, "((%s) pnode->data)", e->parent_etype->ctype);
            buf1 = replace(buf2, "$P", buf);
        } else {
            buf1 = replace(buf2, "$P", "pnode->data");
        }
        xcc_free(buf2);
        fprintf(fp, "            %s\n", buf1);
        xcc_free(buf1);
        n_attributes = xcc_stack_depth(e->attributes);
        
        /* get required attributes and their number */
        for (j = 0; j < n_attributes; j++) {

            xcc_stack_get_data(e->attributes, j, &p);
            a = p;
            
            if (a->required) {
                fprintf(fp, "        attribs_required[%d] = %s;\n",
                    nattribs_required++, print_sharp_name(a->name));
            }
        }
        if (nattribs_required) {
            fprintf(fp, "        nattribs_required = %d;\n\n",
                nattribs_required);
        }
        
        fprintf(fp, "        for (i = 0; attr[i]; i += 2) {\n");
        fprintf(fp, "            askip = 0;\n");
        fprintf(fp, "            aname  = xcc_get_local(attr[i], %s, &askip);\n",
            pns_uri);
        fprintf(fp, "            avalue = attr[i + 1];\n");
        nattribs_required = 0;
        for (j = 0; j < n_attributes; j++) {
            char *pname;

            xcc_stack_get_data(e->attributes, j, &p);
            a = p;

            pname = print_sharp_name(a->name);
            fprintf(fp, "            if (!strcmp(aname, %s)) {\n", pname);
            xcc_free(pname);
            sprintf(abuf, "attribute.%s", a->atype->name);
            buf1 = replace(a->atype->ccode, "$$", abuf);
            buf2 = replace(buf1, "$?", "avalue");
            xcc_free(buf1);
            buf1 = replace(buf2, "$U", "pdata->udata");
            xcc_free(buf2);
            buf2 = replace(buf1, "$0", "xcc_get_root(pdata)");
            xcc_free(buf1);
            fprintf(fp, "                    %s\n", buf2);
            xcc_free(buf2);
            buf1 = replace(a->ccode, "$$", ebuf);
            buf2 = replace(buf1, "$?", abuf);
            xcc_free(buf1);
            buf1 = replace(buf2, "$U", "pdata->udata");
            xcc_free(buf2);
            buf2 = replace(buf1, "$0", "xcc_get_root(pdata)");
            xcc_free(buf1);
            fprintf(fp, "                {\n");
            fprintf(fp, "                        %s\n", buf2);
            fprintf(fp, "                }\n");
            xcc_free(buf2);
            
            if (a->required) {
                /* clear 'required' flag */
                fprintf(fp, "                attribs_required[%d] = NULL;\n",
                    nattribs_required++);
            }
            
            fprintf(fp, "            } else\n");
        }
        fprintf(fp, "            {\n");
        fprintf(fp, "               if (!askip && pdata->exception_handler(XCC_EATTR, aname, el_local, pdata->udata)) {\n");
        fprintf(fp, "                   askip = 1;\n");
        fprintf(fp, "               }\n");
        fprintf(fp, "               if (!askip) {\n");
        fprintf(fp, "                   pdata->error = 1;\n");
        fprintf(fp, "               }\n");
        fprintf(fp, "            }\n");
        fprintf(fp, "            xcc_free(aname);\n");
        fprintf(fp, "        }\n");
        fprintf(fp, "        break;\n");
    }

    xcc_free(pns_uri);

    fprintf(fp, "    }\n\n");

    fprintf(fp, "    if (skip) {\n");
    fprintf(fp, "        element_id = -1;\n");
    fprintf(fp, "    } else {\n");
    fprintf(fp, "        for (i = 0; i < nattribs_required; i++) {\n");
    fprintf(fp, "            aname = attribs_required[i];\n");
    fprintf(fp, "            if (aname) {\n");
    fprintf(fp, "                askip = pdata->exception_handler(XCC_EAREQ, aname, el_local, pdata->udata);\n");
    fprintf(fp, "                if (!askip) {\n");
    fprintf(fp, "                    pdata->error = 1;\n");
    fprintf(fp, "                }\n");
    fprintf(fp, "            }\n");
    fprintf(fp, "        }\n");
    fprintf(fp, "    }\n\n");
    
    fprintf(fp, "    node = xcc_node_new();\n");
    fprintf(fp, "    node->name = el_local;\n");
    fprintf(fp, "    node->id = element_id;\n");
    fprintf(fp, "    node->data = element.unicast;\n");
    fprintf(fp, "    node->occurrence = init_occurrence(element_id);\n");
    
    fprintf(fp, "    xcc_stack_increment(pdata->nodes, node);\n");
    
    fprintf(fp, "}\n\n");
    
    return XCC_RETURN_SUCCESS;
}


static int output_end_handler(const XCC *xcc, FILE *fp)
{
    int i, n_elements;
    char *buf1, *buf2;
    char pbuf[128], ebuf[128];

    n_elements = xcc_stack_depth(xcc->elements);

    fprintf(fp, "static void %s_end_handler(void *data, const char *el)\n",
        xcc->prefix);  
    fprintf(fp, "{\n");
    fprintf(fp, "    XCCParserData *pdata = (XCCParserData *) data;\n");
    fprintf(fp, "    XCCNode *node, *pnode;\n");
    fprintf(fp, "    void *p;\n");
    fprintf(fp, "    int element_id, parent_id, parent_child, skip = 0;\n");

    fprintf(fp, "    XCCEType element, pelement;\n");
    fprintf(fp, "    char *cdata = pdata->cbuffer;\n\n");
    fprintf(fp, "    if (pdata->error) {\n");
    fprintf(fp, "        return;\n");
    fprintf(fp, "    }\n\n");
    fprintf(fp, "    xcc_stack_get_last(pdata->nodes, &p);\n");
    fprintf(fp, "    node = p;\n");
    fprintf(fp, "    element_id = node->id;\n");
    fprintf(fp, "    element.unicast = node->data;\n");

    fprintf(fp, "    switch (element_id) {\n");
    for (i = 0; i < n_elements; i++) {
        Element *e;
        void *p;
        int element_id;

        xcc_stack_get_data(xcc->elements, i, &p);
        e = p;
        if (e->data->s != NULL) {
            sprintf(ebuf, "element.%s", e->etype->name);
            element_id  = e->id;
            fprintf(fp, "    case %d: /* %s */\n", element_id, e->name);
            fprintf(fp, "        {\n");
            buf1 = replace(e->data->s, "$$", ebuf);
            buf2 = replace(buf1, "$?", "cdata");
            fprintf(fp, "            %s\n", buf2);
            xcc_free(buf1);
            xcc_free(buf2);
            fprintf(fp, "        }\n");
            fprintf(fp, "        break;\n");
        }
    }
    fprintf(fp, "    }\n\n");

    fprintf(fp, "    if (node->occurrence) {\n");
    fprintf(fp, "        unsigned int i;\n");
    fprintf(fp, "        for (i = 0; i < %d; i++) {\n", n_elements);
    fprintf(fp, "            if (node->occurrence[i].occurred < node->occurrence[i].minOccurs) {\n");
    fprintf(fp, "                char *cname = get_element_name_by_id(i + %d);\n", KEY_SHIFT);
    fprintf(fp, "                if (!pdata->exception_handler(XCC_EEMIN, cname, node->name, pdata->udata)) {\n");
    fprintf(fp, "                    pdata->error = 1;\n");
    fprintf(fp, "                }\n");
    fprintf(fp, "            }\n");
    fprintf(fp, "        }\n");
    fprintf(fp, "    }\n\n");

    fprintf(fp, "    xcc_stack_decrement(pdata->nodes);\n");

    fprintf(fp, "    if (xcc_stack_depth(pdata->nodes) == 0) {\n");
    fprintf(fp, "        pdata->root = element.unicast;\n");
    fprintf(fp, "        parent_id  = 0;\n");
    fprintf(fp, "        pelement.unicast = NULL;\n");
    fprintf(fp, "    } else {\n");
    fprintf(fp, "        xcc_stack_get_last(pdata->nodes, &p);\n");
    fprintf(fp, "        pnode = p;\n");
    fprintf(fp, "        parent_id  = pnode->id;\n");
    fprintf(fp, "        pelement.unicast = pnode->data;\n");
    fprintf(fp, "    }\n");

    fprintf(fp, "    if (parent_id >= 0 && element_id >= 0) {\n");
    fprintf(fp, "        parent_child = %d*parent_id + element_id;\n", n_elements);
    fprintf(fp, "    } else {\n");
    fprintf(fp, "        parent_child = -1;\n");
    fprintf(fp, "        skip = 1;\n");
    fprintf(fp, "    }\n\n");
    fprintf(fp, "    switch (parent_child) {\n");
    fprintf(fp, "    case 1:\n");
    fprintf(fp, "        break;\n");
    
    for (i = 0; i < n_elements; i++) {
        Element *e;
        void *p;
        int j, n_children, parent_id;
        
        xcc_stack_get_data(xcc->elements, i, &p);
        e = p;
        parent_id  = e->id;
        n_children = xcc_stack_depth(e->children);
        sprintf(pbuf, "pelement.%s", e->etype->name);
        for (j = 0; j < n_children; j++) {
            Child *c;
            Element *ce;
            xcc_stack_get_data(e->children, j, &p);
            c = p;
            ce = get_element_by_name(xcc->elements, c->name);
            if (!ce) {
                xcc_error("couldn't find definition for element %s", c->name);
                return XCC_RETURN_FAILURE;
            }
            sprintf(ebuf, "element.%s", ce->etype->name);
            fprintf(fp, "    case %d:\n", n_elements*parent_id + ce->id);
            fprintf(fp, "        {\n");
            buf1 = replace(c->ccode, "$$", pbuf);
            buf2 = replace(buf1, "$?", ebuf);
            xcc_free(buf1);
            buf1 = replace(buf2, "$U", "pdata->udata");
            xcc_free(buf2);
            buf2 = replace(buf1, "$0", "xcc_get_root(pdata)");
            xcc_free(buf1);
            fprintf(fp, "        %s\n", buf2);
            xcc_free(buf2);
            fprintf(fp, "        }\n");
            fprintf(fp, "        break;\n");
        }
    }

    fprintf(fp, "    default:\n");
    fprintf(fp, "        if (!skip) {\n");
    fprintf(fp, "            pdata->exception_handler(XCC_EINTR, NULL, NULL, pdata->udata);\n");
    fprintf(fp, "            pdata->error = 1;\n");
    fprintf(fp, "        }\n");
    fprintf(fp, "        break;\n");
    fprintf(fp, "    }\n\n");

    fprintf(fp, "    pdata->cbuflen = 0;\n");
    fprintf(fp, "    if (pdata->cbufsize) {\n");
    fprintf(fp, "        pdata->cbuffer[0] = '\\0';\n");
    fprintf(fp, "    }\n");
    fprintf(fp, "}\n\n");
    
    return XCC_RETURN_SUCCESS;
}

static int output_parser(const XCC *xcc, FILE *fp)
{
    fprintf(fp, "int %s_parse(FILE *fp, void **root, void *udata, XCCExceptionHandler exception_handler)\n", xcc->prefix);
    fprintf(fp, "{\n");

    fprintf(fp, "    if (xcc_run(fp, root, udata, %s_start_handler, %s_end_handler, exception_handler)\n",
        xcc->prefix, xcc->prefix);
    fprintf(fp, "        != XCC_RETURN_SUCCESS) {\n");
    fprintf(fp, "        return XCC_RETURN_FAILURE;\n");
    fprintf(fp, "    }\n");
    
    fprintf(fp, "    return XCC_RETURN_SUCCESS;\n");
    fprintf(fp, "}\n");

    return XCC_RETURN_SUCCESS;
}


int xcc_output_parser(const XCC *xcc, FILE *fp, int bundle)
{
    if (bundle) {
        output_bundle(fp);
    } else {
        output_header(fp);
    }
    
    output_preamble(xcc, fp);
    
    output_atype_union(xcc, fp);
    output_etype_union(xcc, fp);

    /* TODO: sort elements ?? */

    output_element_tab(xcc, fp);

    output_init_occurrence(xcc, fp);

    output_start_handler(xcc, fp);
    
    output_end_handler(xcc, fp);

    output_parser(xcc, fp);

    output_postamble(xcc, fp);

    return XCC_RETURN_SUCCESS;
}

int xcc_output_schema(const XCC *xcc, FILE *fp)
{
    XFile *xf = xfile_new(fp);
    Attributes *attrs = attributes_new();
    int i, n_elements;
    char buf[128];

    if (!xf || !attrs) {
        return XCC_RETURN_FAILURE;
    }
    
    attributes_reset(attrs);

    xfile_set_ns(xf, "xs", "http://www.w3.org/2001/XMLSchema", 1);

    xfile_begin(xf, 1, NULL, NULL, "schema", attrs);

    sprintf(buf, "Generated by %s", xcc_get_version_string());
    xfile_comment(xf, buf);

    n_elements = xcc_stack_depth(xcc->elements);
    for (i = 0; i < n_elements; i++) {
        Element *e;
        void *p;
        int j, n_children, n_attributes, parent_id;
        
        xcc_stack_get_data(xcc->elements, i, &p);
        e = p;
        parent_id  = e->id;
        
        n_children = xcc_stack_depth(e->children);
        n_attributes = xcc_stack_depth(e->attributes);

        attributes_reset(attrs);
        attributes_set_sval(attrs, "name", e->name);
        
        if (n_children || n_attributes) {
            xfile_begin_element(xf, "element", attrs);
            
            attributes_reset(attrs);
            if (e->data->s) {
                attributes_set_sval(attrs, "mixed", "true");
            }
            xfile_begin_element(xf, "complexType", attrs);
            
            if (n_children) {
                xfile_begin_element(xf, "sequence", NULL);
                for (j = 0; j < n_children; j++) {
                    Child *c;
                    xcc_stack_get_data(e->children, j, &p);
                    c = p;

                    attributes_reset(attrs);
                    attributes_set_sval(attrs, "ref", c->name);
                    attributes_set_ival(attrs, "minOccurs", c->minOccurs);
                    if (c->maxOccurs) {
                        attributes_set_ival(attrs, "maxOccurs", c->maxOccurs);
                    } else {
                        attributes_set_sval(attrs, "maxOccurs", "unbounded");
                    }
                    xfile_empty_element(xf, "element", attrs);
                }
                xfile_end_element(xf, "sequence");
            }
            
            for (j = 0; j < n_attributes; j++) {
                Attribute *a;

                xcc_stack_get_data(e->attributes, j, &p);
                a = p;

                attributes_reset(attrs);
                attributes_set_sval(attrs, "name", a->name);
                if (a->required) {
                    attributes_set_sval(attrs, "use", "required");
                }
                xfile_empty_element(xf, "attribute", attrs);
            }

            xfile_end_element(xf, "complexType");
            
            xfile_end_element(xf, "element");
        } else {
            xfile_empty_element(xf, "element", attrs);
        }

    }

    attributes_free(attrs);
    
    xfile_end(xf);
    xfile_free(xf);

    return XCC_RETURN_SUCCESS;
}

static void usage(const char *arg0, FILE *fp)
{
    fprintf(fp, "Usage: %s [options]\n", arg0);
    fprintf(fp, "Available options:\n");
    fprintf(fp, "  -i <file>  input file [stdin]\n");
    fprintf(fp, "  -o <file>  output file [stdout]\n");
    fprintf(fp, "  -b         bundle auxiliary stuff into parser\n");
    fprintf(fp, "  -s         output WXS schema\n");
    fprintf(fp, "  -V         display version info and exit\n");
    fprintf(fp, "  -h         print this help and exit\n");
}

int xcc_parse_opts(XCCOpts *xopts, int argc, char * const argv[])
{
    int opt;

    while ((opt = getopt(argc, argv, "i:o:bsVh")) != -1) {
        switch (opt) {
        case 'i':
            xopts->ifile = optarg;
            break;
        case 'o':
            xopts->ofile = optarg;
            break;
        case 'b':
            xopts->bundle = 1;
            break;
        case 's':
            xopts->schema = 1;
            break;
        case 'V':
            printf("%s\n", xcc_get_version_string());
            exit(0);
            break;
        case 'h':
            usage(argv[0], stdout);
            exit(0);
            break;
        default:
            usage(argv[0], stderr);
            return XCC_RETURN_FAILURE;
            break;
        }
    }

    if (!xopts->ifile) {
        xopts->ifp = stdin;
    } else {
        xopts->ifp = fopen(xopts->ifile, "r");
    }
    if (!xopts->ifp) {
        xcc_error("can't open input stream");
        return XCC_RETURN_FAILURE;
    }
    
    if (!xopts->ofile) {
        xopts->ofp = stdout;
    } else {
        xopts->ofp = fopen(xopts->ofile, "wb");
    }
    if (!xopts->ofp) {
        xcc_error("can't open output stream");
        return XCC_RETURN_FAILURE;
    }
    
    return XCC_RETURN_SUCCESS;
}
